/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2015 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.commerce.api.classification;

import java.util.List;

import aQute.bnd.annotation.ProviderType;
import com.adobe.cq.commerce.api.Product;

/**
 * The <code>ClassificationCategory<code> is the building unit of a <code>Classification<code> as a classification
 * is made of categories arranged in a tree-like structure.
 *
 * A category:
 * - has a parent category, except for the root category which is the root of the category tree.
 * - can have sub categories.
 * - can have one or more <code>ClassificationAttribute</code>s.
 * - can have <code>Product</code>s assigned to it.
 *
 * Since 6.2
 */
@ProviderType
public interface ClassificationCategory {

    /**
     * Returns the title of the category.
     *
     * @return title of the category.
     */
    public String getTitle();

    /**
     * Returns the description of the category.
     *
     * @return description of the category.
     */
    public String getDescription();

    /**
     * Returns the path of the category if represented in the repository. Otherwise, it returns an ID
     * uniquely identifying the category to the external commerce engine.
     *
     * @return path/ID of the category.
     */
    public String getPath();

    /**
     * Returns the classification which this category belongs to.
     *
     * @return classification which this category belongs to.
     */
    public Classification getClassification();

    /**
     * Returns true if the category is the root category of the classification.
     *
     * @return true if the category is the root category of the classification.
     *         false otherwise
     */
    public boolean isRootCategory();

    /**
     * Returns an ordered list of attributes that are directly defined for this category.
     *
     * To get all the attributes (those defined directly for this category and those defined for the ancestor
     * categories), use {@link ClassificationCategory#getAttributes}
     *
     * @return the list of direct attributes.
     */
    public List<ClassificationAttribute> getDirectAttributes();

    /**
     * Returns an ordered list of all the attributes by taking the direct attributes of this category and those
     * defined for the ancestor categories.
     *
     * The list is ordered as follows: first the direct attributes, then the attributes from the parent category,
     * and so on.
     *
     * @return the list of inherited attributes.
     */
    public List<ClassificationAttribute> getAttributes();

    /**
     * Returns the classification attribute defined by the given name.
     *
     * @param name The attribute name
     * @return The {@link ClassificationAttribute} or null if it does not exist
     */
    public ClassificationAttribute getAttribute(String name);

    /**
     * Returns the parent category.
     *
     * @return the parent category.
     */
    public ClassificationCategory getParentCategory();

    /**
     * Returns the sub categories.
     *
     * @return the sub categories.
     */
    public List<ClassificationCategory> getSubCategories();

    /**
     * Returns the classification category defined by the given name.
     *
     * @param name The category name
     * @return The {@link ClassificationCategory} or null if it does not exist
     */
    public ClassificationCategory getSubCategory(String name);

    /**
     * Returns an iterable of products that are directly assigned to this category.
     *
     * @return iterable of the directly assigned products.
     */
    public Iterable<Product> getDirectProducts();

    /**
     * Returns an iterable of the products belonging to this category.
     *
     * If products are directly assigned to this category, an iterable of those products is returned.
     * Otherwise, this method is applied to all the sub categories and an iterable of the generated list is returned.
     *
     * @return iterable of the belonging products.
     */
    public Iterable<Product> getProducts();

    /**
     * Returns true if the product is directly assigned to the category.
     *
     * @param product product to be checked
     * @return true if product is directly assigned to the category.
     *         false otherwise
     */
    public boolean hasDirectProduct(Product product);

    /**
     * Returns true if the product belongs to the category.
     *
     * If products are directly assigned to this category, those products are returned.
     * Otherwise, this method is applied to all the sub categories.
     *
     * @param product product to be checked
     * @return true if product belongs to the category.
     *         false otherwise
     */
    public boolean hasProduct(Product product);

}
