/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2015 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.commerce.api.classification;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import aQute.bnd.annotation.ProviderType;
import com.adobe.cq.commerce.api.CommerceException;
import com.adobe.cq.commerce.api.Product;

/**
 * The <code>ClassificationManager</code> provides methods to retrieve, update, create and
 * update product classifications.
 *
 * Since 6.2
 */
@ProviderType
public interface ClassificationManager {

    /**
     * Returns a product classification for the given <code>path</code> or an ID
     * uniquely identifying the {@link Classification} to the external commerce engine.
     *
     * It returns null if given path/ID is not a product classification or is not defined
     *
     * @param path path/ID that represents the classification
     * @return The {@link Classification} representing the classification.
     *
     */
    public Classification getClassification(String path);

    /**
     * Returns a classification category for the given <code>path</code> or an ID
     * uniquely identifying the {@link ClassificationCategory} to the external commerce engine.
     *
     * It returns null if given path/ID is not a classification category or is not defined
     *
     * @param path path/ID that represents the category
     * @return The {@link ClassificationCategory} representing the category.
     *
     */
    public ClassificationCategory getCategory(String path);

    /**
     * Returns a classification attribute for the given <code>path</code> or an ID
     * uniquely identifying the {@link ClassificationAttribute} to the external commerce engine.
     *
     * It returns null if given path/ID is not a classification attribute or is not defined
     *
     * @param path path/ID that represents the attribute
     * @return The {@link ClassificationAttribute} representing the attribute.
     *
     */
    public ClassificationAttribute getAttribute(String path);

    /**
     * Creates a product classification with a given name under the <code>parentPath</code>.
     *
     * @param parentPath parent path where classification needs to be created.
     * @param name The name for classification.
     * @param properties The additional data for product classification
     *
     * @return The {@link Classification} representing the created classification.
     *
     * @throws CommerceException if the classification was not created
     */
    public Classification createClassification(String parentPath, String name, Map<String, Object> properties) throws CommerceException;

    /**
     * Removes the classification.
     *
     * @param path path/ID that represents the classification
     *
     * @throws CommerceException if the classification was not removed
     */
    void removeClassification(String path) throws CommerceException;

    /**
     * Updates the classification.
     *
     * @param path path/ID that represents the classification
     * @param properties The additional data for product classification
     *
     * @throws CommerceException if the classification was not updated
     */
    void updateClassification(String path, Map<String, Object> properties) throws CommerceException;

    /**
     * Adds a root category to the classification.
     *
     * The root category is the base category of the classification and does not have a parent category.
     *
     * @param classificationPath path/ID that represents the classification
     * @param properties The additional data for the root category
     *
     * @return The {@link ClassificationCategory} representing the created category.
     *
     * @throws CommerceException if the root category was not created
     *                           or if the classification already has a root category
     */
    public ClassificationCategory addRootCategory(String classificationPath, Map<String, Object> properties) throws CommerceException;

    /**
     * Adds a sub category to the specified category.
     *
     * @param parentPath parent path / ID that represents the parent category
     * @param properties The additional data for the category
     *
     * @return The {@link ClassificationCategory} representing the created category.
     *
     * @throws CommerceException if the category was not created
     */
    public ClassificationCategory addCategory(String parentPath, String name, Map<String, Object> properties) throws CommerceException;

    /**
     * Removes the category.
     *
     * @param path path/ID that represents the category
     *
     * @throws CommerceException if the category was not removed
     */
    public void removeCategory(String path) throws CommerceException;

    /**
     * Updates the category.
     *
     * @param path path/ID that represents the category
     * @param properties The additional data for the category
     *
     * @throws CommerceException if the category was not updated
     */
    public void updateCategory(String path, Map<String, Object> properties) throws CommerceException;

    /**
     * Adds an attribute to the specified category.
     *
     * @param categoryPath path/ID that represents the category
     * @param properties The additional data for the attribute
     *
     * @return The {@link ClassificationAttribute} representing the created attribute.
     *
     * @throws CommerceException if the attribute was not created
     */
    public ClassificationAttribute addAttribute(String categoryPath, String name, Map<String, Object> properties) throws CommerceException;

    /**
     * Removes the attribute.
     *
     * @param path path/ID that represents the attribute
     *
     * @throws CommerceException if the attribute was not removed
     */
    public void removeAttribute(String path) throws CommerceException;

    /**
     * Updates the attribute.
     *
     * @param path path/ID that represents the attribute
     * @param properties The additional data for the attribute
     *
     * @throws CommerceException if the attribute was not updated
     */
    public void updateAttribute(String path, Map<String, Object> properties) throws CommerceException;

    /**
     * Inserts the referenced attribute <code>srcAttributePath</code> into the list of attributes at the position
     * immediately before the referenced attribute <code>destAttributePath</code>.
     *
     * To insert the referenced attribute into last position, <code>destAttributePath</code> can be null.
     *
     * @param srcAttributePath Referenced attribute path/ID that needs to be moved in the order
     * @param destAttributePath Referenced attribute path/ID before which the <code>srcAttributePath</code> will be placed.
     *
     * @throws CommerceException if the operation fails
     */
    public void orderBefore(String srcAttributePath, String destAttributePath) throws CommerceException;

    /**
     * Assigns products to the category.
     *
     * @param categoryPath path/ID that represents the category
     * @param productListDefinition A map of properties (key-value pairs) defining a set of products
     *                              (can be used e.g. to define a query or a tag-based search)
     *
     * @throws CommerceException if the operation failed
     */
    public void assignProducts(String categoryPath, Map<String, Object> productListDefinition) throws CommerceException;

    /**
     * Retrieves the classifications in which the product is assigned.
     *
     * @param product The product
     *
     * @return A list of {@link Classification}s in which the product is assigned.
     */
    public Collection<Classification> getClassifications(Product product);

    /**
     * Retrieves the classification categories in which the product is assigned.
     *
     * @param product The product
     *
     * @return A list of {@link ClassificationCategory}s in which the product is assigned.
     */
    public Collection<ClassificationCategory> getClassificationCategories(Product product);

    /**
     * Retrieves the classifications in which at least one of the product is assigned.
     *
     * @param products The list of products
     *
     * @return A list of {@link Classification}s in which the products are assigned.
     */
    public Collection<Classification> getClassifications(List<Product> products);

    /**
     * Retrieves the classification categories in which at least one of the product is assigned.
     *
     * @param products The list of products
     *
     * @return A list of {@link ClassificationCategory}s in which the products are assigned.
     */
    public Collection<ClassificationCategory> getClassificationCategories(List<Product> products);

}
