/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.commerce.api.collection;

import java.util.Iterator;
import java.util.Map;

import aQute.bnd.annotation.ProviderType;
import com.adobe.cq.commerce.api.CommerceException;
import com.adobe.cq.commerce.api.Product;

/**
 * ProductCollection is an ordered collection of either product or product collection references.
 * The collection does not store the actual products/collections, it only points to them.
 *
 */
@ProviderType
public interface ProductCollection {

    /**
     * Returns the title of the collection.
     *
     * @return title of the collection.
     */
    public String getTitle();

    /**
     * Returns the description of the collection.
     *
     * @return description of the collection.
     */
    public String getDescription();

    /**
     * Returns the type of the collection.
     *
     * @return type of the collection.
     */
    public String getType();

    /**
     * Returns the path of the collection if represented in the repository.  Otherwise, it returns an ID
     * uniquely identifying the collection to the external commerce engine.
     *
     * @return path/ID of the collection.
     */
    public String getPath();

    /**
     * Returns the collection properties.
     *
     * @return The collection properties as an unmodifiable {@link Map}.
     */
    public Map<String, Object> getProperties();

    /**
     * Returns a named collection property converted to a given type.
     *
     * @param key The name of the property
     * @param type The type to which the property should be converted.
     * @return The named value converted to type T.
     */
    public <T> T getProperty(String key, Class<T> type);

    /**
     * Returns a named collection property or a default value if the property can not be retrieved.
     *
     * @param key The name of the property
     * @param defaultValue The default value to use if the named property does
     *            not exist or cannot be converted to the requested type. The
     *            default value is also used to define the type to convert the
     *            value to. If this is <code>null</code> any existing property is
     *            not converted.
     * @return The named value converted to type T or the default value if
     *         non existing or can't be converted.
     */
    public <T> T getProperty(String key, T defaultValue);

    /**
     * Sets the collection properties. New properties are added, existing ones are replaced.
     *
     * @param properties The new collection properties.
     * @throws CommerceException if setting the properties fails.
     */
    public void setProperties(Map<String, Object> properties) throws CommerceException;

    /**
     * Sets a collection property.
     *
     * @param key The name of the property
     * @param value The value of the property
     * @throws CommerceException if setting the properties fails.
     */
    public void setProperty(String key, Object value) throws CommerceException;

    /**
     * Returns an iterator over product or collection paths/IDs directly referenced in the collection.
     * These are the first level descendants of the collection as opposed to all levels descendant
     * products returned by {@link ProductCollection#getProducts}
     *
     * @return iterator over product or collection paths/IDs directly referred in collection.
     */
    public Iterator<String> getDirectReferences();

    /**
     * Returns true if the product or the collection referenced by the path/ID is directly referenced in the collection.
     * To check if a product is part of all levels descendants of the collection, use {@link ProductCollection#contains}.
     *
     * @param path product or collection path to be added, if the product/collection is represented in the repository.
     *             Otherwise, it is an ID uniquely identifying the product/collection to the external commerce engine.
     *
     * @return true if product or collection path is part of the collection.
     *         false otherwise
     */
    public boolean hasDirectReference(String path);

    /**
     * Returns an iterator over products referred in the collection in the specified order.
     *
     * @return iterator over products referred in collection.
     */
    public Iterator<Product> getProducts();

    /**
     * Returns true if product reference is part of the collection.
     *
     * @param product product to be checked
     * @return true if product is part of the collection.
     *         false otherwise
     */
    public boolean contains(Product product);

    /**
     * Creates a new entry (product or collection) in the collection at the last position
     *
     * @param path product or collection path to be added, if the product/collection is represented in the repository.
     *             Otherwise, it is an ID uniquely identifying the product/collection to the external commerce engine.
     *
     * @throws CommerceException if the operation fails, if collection already contained
     *          the product/collection or product/collection is null.
     */
    public void add(String path) throws CommerceException;

    /**
     * Removes an entry of product or collection from collection.
     *
     * @param path product or collection path to be added, if the product/collection is represented in the repository.
     *             Otherwise, it is an ID uniquely identifying the product/collection to the external commerce engine.
     *
     * @throws CommerceException if the operation fails, or if product/collection reference was
     *          not removed / not present
     */
    public void remove(String path) throws CommerceException;

    /**
     * Inserts the referenced product or collection <code>srcPath</code>
     * into the collection entry at the position immediately before the referenced product
     * or collection <code>destPath</code>.
     *
     * To insert the referenced product/collection into last position, <code>destPath</code> can be null.
     *
     * @param srcPath Referenced product/collection path/ID that needs to be moved in the order
     * @param destPath Referenced product/collection path/ID before which the <code>srcPath</code> will be placed.
     *
     * @throws CommerceException if the operation fails
     */
    public void orderBefore(String srcPath, String destPath) throws CommerceException;

}
