/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.commerce.api.collection;

import java.util.Iterator;
import java.util.Map;

import aQute.bnd.annotation.ProviderType;
import com.adobe.cq.commerce.api.CommerceException;
import com.adobe.cq.commerce.api.Product;

/**
 * The <code>ProductCollectionManager<code> defines the API to get, create and delete
 * product collections {@link ProductCollection}.
 *
 * A ProductCollectionManager can be adapted from a {@link org.apache.sling.api.resource.ResourceResolver}.
 *
 */
@ProviderType
public interface ProductCollectionManager {

    /**
     * Returns the product collections that reference a <code>product</code>.
     *
     * @param product
     * @return An {@link java.util.Iterator} over the {@link ProductCollection}s referencing the product.
     *
     */
    Iterator<ProductCollection> getProductCollections(Product product);

    /**
     * Returns a product collection for the given <code>path</code> or an ID
     * uniquely identifying the {@link ProductCollection} to the external commerce engine.
     *
     * It returns null if given path/ID is not a product collection or is not defined
     *
     * @param path path/ID that represents the collection
     * @return The {@link ProductCollection} representing the collection.
     *
     */
    ProductCollection getCollection(String path);

    /**
     * Creates a product collection with a given name under the <code>parentPath</code>.
     *
     * @param parentPath parent path where collection needs to be created.
     * @param name The name for collection.
     *
     * @return The {@link ProductCollection} representing the created collection.
     *
     * @throws CommerceException if the collection was not created
     */
    ProductCollection createCollection(String parentPath, String name) throws CommerceException;

    /**
     * This method creates a product collection with a given name under the <code>parentPath</code>.
     *
     * @param parentPath parent path where collection needs to be created.
     * @param name The name for collection.
     * @param properties The additional data for product collection
     *
     * @return The {@link ProductCollection} representing the created collection.
     *
     * @throws CommerceException if the collection was not created
     */
    ProductCollection createCollection(String parentPath, String name, Map<String,Object> properties) throws CommerceException;

    /**
     * Updates the {@link ProductCollection}.
     *
     * @param collection the collection to be updated.
     * @param properties The additional data for product collection
     *
     * @throws CommerceException if the collection was not updated
     */
    void updateCollection(ProductCollection collection, Map<String,Object> properties) throws CommerceException;

    /**
     * Removes the {@link ProductCollection}.
     *
     * @param collection the collection to be deleted.
     *
     * @throws CommerceException if the collection was not deleted
     */
    void deleteCollection(ProductCollection collection) throws CommerceException;
}