/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.commerce.api.promotion;

import aQute.bnd.annotation.ConsumerType;
import org.apache.sling.api.SlingHttpServletRequest;

import com.adobe.cq.commerce.api.CommerceException;
import com.adobe.cq.commerce.api.CommerceSession;
import com.adobe.cq.commerce.api.PriceInfo;
import com.adobe.cq.commerce.api.CommerceSession.CartEntry;

import java.util.Map;

/**
 * The <code>PromotionHandler</code> interface is implemented by services which provide business logic
 * for {@link Promotion}s.
 *
 * <p>Services should define a property <code>PromotionHandler.TYPE</code> with a unique identifier,
 * which is then used by individual promotions to select a particular PromotionHandler.</p?
 * 
 * @since 5.6
 */
@ConsumerType
public interface PromotionHandler {

    /**
     * Service property name.
     */
    public static final String TYPE = "commerce.promotion.type";

    /**
     * Apply a promotion to a cart line item. The method returns a discount {@link PriceInfo} instance
     * or <code>null</code> if no discount was applied.
     * @param commerceSession The commerce session containing the cart.
     * @param promotion The {@link Promotion} to apply.
     * @param cartEntry The cart line item within the session's cart.
     * @return the discount amount (as a <code>PriceInfo</code>), or <code>null</code>.
     */
    public PriceInfo applyCartEntryPromotion(CommerceSession commerceSession, Promotion promotion, CartEntry cartEntry)
            throws CommerceException;

    /**
     * Apply a promotion to an order. The method returns a discount {@link PriceInfo} instance or
     * <code>null</code> if no discount was applied.
     * @param commerceSession The commerce session containing the order.
     * @param promotion The {@link Promotion} to apply.
     * @return the discount amount (as a <code>PriceInfo</code>), or <code>null</code>.
     */
    public PriceInfo applyOrderPromotion(CommerceSession commerceSession, Promotion promotion) throws CommerceException;

    /**
     * Apply a promotion to an order's shipping amount. The method returns a discount
     * <code>PriceInfo</code> instance or <code>null</code> if no discount was applied.
     * @param commerceSession The commerce session containing the order.
     * @param promotion The {@link Promotion} to apply.
     * @return the shipping discount amount (as a <code>PriceInfo</code>), or <code>null</code>.
     */
    public PriceInfo applyShippingPromotion(CommerceSession commerceSession, Promotion promotion) throws CommerceException;

    /**
     * @deprecated since 5.6.200; use {@link #getDescription(org.apache.sling.api.SlingHttpServletRequest,
     * CommerceSession, Promotion)} instead.
     */
    @Deprecated
    public String getMessage(SlingHttpServletRequest request, CommerceSession commerceSession, Promotion promotion)
            throws CommerceException;

    /**
     * Allows a promotion handler to define a custom, author-oriented message for a promotion.
     *
     * <p>The <code>info.geometrixx.commons.impl.PerfectPartnerPromotionHandler</code>, for
     * instance, uses this to list the qualifying pairs of products in the current cart.</p>
     *
     * @param request The current request (used for access-control).
     * @param commerceSession The current {@link CommerceSession}.
     * @param promotion The {@link Promotion}.
     * @return a custom, author-oriented message to display.
     * @since 5.6.200
     */
    public String getDescription(SlingHttpServletRequest request, CommerceSession commerceSession, Promotion promotion)
            throws CommerceException;

    /**
     * Allows a promotion handler to define shopper-oriented messages for a promotion.
     *
     * <p>An order-specific promotion should return a single message with a <code>-1</code> key.</p>
     *
     * <p>A cart-entry-specific promotion should return its message(s) keyed to the cart entries they
     * apply to.</p>
     *
     * @param request The current request (used for access-control).
     * @param commerceSession The current {@link CommerceSession}.
     * @param promotion The {@link Promotion}.
     * @return A list of messages, either keyed to individual cart entry indexes, or keyed to the
     *         order (<code>-1</code>).
     * @throws CommerceException
     * @since 5.6.200
     */
    public Map<Integer, String> getMessages(SlingHttpServletRequest request, CommerceSession commerceSession, Promotion promotion)
            throws CommerceException;

    /**
     * Informs the promotion handler that something under the promotions root has been edited, and the
     * handler should invalidate any caches it might be keeping.
     */
    public void invalidateCaches();
}
