/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2015 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.cq.commerce.api.smartlist;

import aQute.bnd.annotation.ProviderType;
import com.adobe.cq.commerce.api.CommerceException;
import org.apache.commons.collections.Predicate;

import java.util.List;
import java.util.Map;

/**
 * The <code>SmartListManager<code> defines the API to get, create and delete smart lists {@link SmartList} and methods
 * related to sharing of a smart list.
 *
 * A SmartListManager can be retrieved from the {@link com.adobe.cq.commerce.api.CommerceSession}.
 */
@ProviderType
public interface SmartListManager {

    /**
     * Returns a smart list for the given <code>path</code> or an ID uniquely identifying the {@link SmartList} to the
     * external commerce engine. It returns null if given path/ID is not a smart list or is not defined.
     *
     * @param path The path/ID that represents the smart list
     * @return The {@link SmartList} representing the smart list or <code>null</code> if not found.
     */
    SmartList getSmartList(String path);

    /**
     * Get a list of smart lists, based on the provided filter, this can be all, only personal lists or lists shared by
     * friends.
     *
     * @param filter An optional implementation-specific predicate name. For example to filter personal lists and lists
     *               shared by friends.
     * @return a list of {@code SmartList} objects.
     */
    List<SmartList> getSmartLists(String filter);

    /**
     * Creates a smart list with a given name.
     *
     * @param name The name of the smart list.
     * @return The {@link SmartList} representing the created smart list.
     * @throws CommerceException if the smart list was not created
     */
    SmartList createSmartList(String name) throws CommerceException;

    /**
     * Creates a smart list with a given name.
     *
     * @param name       The name of the smart list.
     * @param properties The additional data like description for the smart list.
     * @return The {@link SmartList} representing the created smart list.
     * @throws CommerceException if the smart list was not created
     */
    SmartList createSmartList(String name, Map<String, Object> properties) throws CommerceException;

    /**
     * Updates the {@link SmartList}.
     *
     * @param path The path/ID that represents the smart list to be updated.
     * @param properties The additional data for smart list like title or description
     * @throws CommerceException if the smart list was not updated
     */
    void updateSmartList(String path, Map<String, Object> properties) throws CommerceException;

    /**
     * Removes the {@link SmartList}.
     *
     * @param path The path/ID that represents the smart list to be deleted.
     * @throws CommerceException if the smart list was not deleted
     */
    void deleteSmartList(String path) throws CommerceException;

    /**
     * Mark the current smart list as default smart list. This implies, depending on the implementation, all other smart
     * list of the same owner are marked as non default.
     *
     * @param path The path/ID that represents the smart list to be marked as default.
     * @throws CommerceException if the smart list was not marked as default
     */
    void makeSmartListDefault(String path) throws CommerceException;

    /**
     * Share the current smart list with third party customers. Sharing information (recipients, sharing state, messages)
     * are provided via the properties parameter. Use <code>null</code> for properties to "unshare" a smart-list and
     * set it back to personal.
     *
     * @param path The path/ID that represents the smart list to be shared.
     * @param properties The sharing properties.
     * @throws CommerceException if the smart list was not marked as default
     */
    void shareSmartList(String path, Map<String, Object> properties) throws CommerceException;
}
