/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.commerce.common;

import org.apache.commons.codec.net.URLCodec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * The <code>CookieUtil</code> helps in working with cookies on request/response objects.
 */
public class CookieUtil {

    private static final Logger log = LoggerFactory.getLogger(CookieUtil.class);

    private static final URLCodec urlCodec = new URLCodec();

    private static final String HTTP_HEADER_SET_COOKIE = "Set-Cookie";
    private static final String URL_HOST_REGEX = "https?://([\\w\\d_\\.\\-]*)(:\\d+)?(/.*)?";

    public static final String SESSION_COOKIE = "JSESSIONID";

    public static final boolean HTTP_ONLY = true;   // self-documenting value for the httpOnly parameter

    /**
     * Sets a cookie into the current response.
     * @param request   The current request.
     * @param response  The current response.
     * @param name      The cookie name.
     * @param value     The cookie value.
     * @param maxAge    The expiry (in seconds from now).  Pass -1 for no expiry, 0 to remove the cookie immediately.
     * @param httpOnly  Indicates the cookie should be accessible only from the server.
     */
    public static void setCookie(HttpServletRequest request, HttpServletResponse response, String name, String value, int maxAge,
                                 boolean httpOnly) {
        /*
         * The Servlet Spec 2.5 does not allow us to set the commonly used HttpOnly attribute
         * on cookies (Servlet API 3.0 does) so we create the Set-Cookie header manually.
         * See http://www.owasp.org/index.php/HttpOnly for information on the HttpOnly attribute.
         */
        final StringBuilder header = new StringBuilder();

        header.append(name).append("=").append(value);

        final String contextPath = request.getContextPath();
        final String cookiePath = (contextPath == null || contextPath.length() == 0) ? "/" : contextPath;
        header.append("; Path=").append(cookiePath);

        if (maxAge >= 0) {
            header.append("; Max-Age=").append(maxAge);
        }

        if (httpOnly) {
            header.append("; HttpOnly");
        }

        // Always set the secure flag from an https request
        if (request.isSecure()) {
            header.append("; Secure");
        }

        response.addHeader(HTTP_HEADER_SET_COOKIE, header.toString());
    }

    /**
     * Extracts the host from a given URL. This utility method can be used for creating cookies to be set on requests
     * to a remote server.
     * @param url   A URL
     * @return      Host part of URL or <code>null</code> for an invalid URL.
     */
    public static String hostFromUrl(String url) {
        Matcher matcher = Pattern.compile(URL_HOST_REGEX).matcher(url);
        if (matcher.matches()) {
            return matcher.group(1);
        }
        return null;
    }

    /*
     * ==================================================================================================
     * Legacy support for Apache httpClient 3.x.
     *
     * For backwards-compatibility only.  Apache httpClient 3.x has been EOLed and contains several
     * security vulnerabilities:
     *
     * http://web.nvd.nist.gov/view/vuln/detail?vulnId=CVE-2012-5783
     * http://seclists.org/oss-sec/2013/q1/286
     * ==================================================================================================
     */

    /**
     * Decodes and returns cookies with defined session prefix.
     * @param request   The request from which to read the cookies.
     * @param prefix    A name prefix filter.
     * @return          A <code>List</code> of cookies matching the prefix filter.
     *
     * @deprecated since 6.1; Apache httpClient 3.x has been EOLed and contains security vulnerabilities
     */
    @Deprecated
    public static List<org.apache.commons.httpclient.Cookie> getSessionCookies(HttpServletRequest request, String prefix, String remoteHost) {
        Cookie[] requestCookies = request.getCookies();
        List<org.apache.commons.httpclient.Cookie> cookies = new ArrayList<org.apache.commons.httpclient.Cookie>();

        try {
            if (requestCookies != null) {
                // looks for session cookies on request
                for (javax.servlet.http.Cookie c : requestCookies) {
                    if (c.getName().startsWith(prefix)) {
                        String[] values = urlCodec.decode(c.getValue()).split(";");
                        cookies.add(new org.apache.commons.httpclient.Cookie(remoteHost, SESSION_COOKIE, values[0], values[1],
                                c.getMaxAge(), c.getSecure()));
                    }
                }
            }
        } catch (Exception e) {
            log.error("Could not get jcrSession cookies from request: ", e);
            return new ArrayList<org.apache.commons.httpclient.Cookie>();
        }

        return cookies;
    }

    /**
     * Encodes and sets the provided cookies on the response.
     * @param response  The response to which the cookies are to be written.
     * @param prefix    A string to be prefixed to each cookie name.
     * @param cookies   The list of cookies.
     *
     * @deprecated since 6.1; Apache httpClient 3.x has been EOLed and contains security vulnerabilities
     */
    @Deprecated
    public static void setSessionCookies(HttpServletResponse response, String prefix, List<org.apache.commons.httpclient.Cookie> cookies) {
        try {
            for (org.apache.commons.httpclient.Cookie c : cookies) {
                String value = urlCodec.encode(c.getValue() + ";" + c.getPath());
                javax.servlet.http.Cookie cookie = new javax.servlet.http.Cookie(prefix + c.getPath().substring(1), value);
                cookie.setPath("/");

                response.addCookie(cookie);
            }
        } catch (Exception e) {
            log.error("Could not set jcrSession cookies on response: ", e);
        }
    }

    /**
     * Checks if there are any session cookies in the HTTP method's response.
     * @param method    The executed HTTP method.
     * @return          <code>true</code> if response contains session cookies. <code>false</code> otherwise.
     *
     * @deprecated since 6.1; Apache httpClient 3.x has been EOLed and contains security vulnerabilities
     */
    @Deprecated
    public static boolean hasUpdatedSessionCookies(org.apache.commons.httpclient.HttpMethod method) {
        if (method.hasBeenUsed()) {
            org.apache.commons.httpclient.Header[] headers = method.getResponseHeaders(HTTP_HEADER_SET_COOKIE);

            for (int i = 0; i < headers.length; ++i) {
                if (headers[i].getValue().contains(SESSION_COOKIE)) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * Checks to see if there is a session cookie in the list.
     * @param cookies   The list of cookies.
     * @return          <code>true</code> if <code>cookies</code> contains a session cookie.
     *
     * @deprecated since 6.1; Apache httpClient 3.x has been EOLed and contains security vulnerabilities
     */
    @Deprecated
    public static boolean hasSessionCookie(List<org.apache.commons.httpclient.Cookie> cookies) {
        for (org.apache.commons.httpclient.Cookie c : cookies) {
            if (c.getName().equals(SESSION_COOKIE)) {
                return true;
            }
        }
        return false;
    }
}
