/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.commerce.pim.common;

import java.io.IOException;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletResponse;

import aQute.bnd.annotation.ConsumerType;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Service;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.SlingHttpServletResponse;
import org.apache.sling.api.resource.ResourceResolver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.cq.commerce.pim.api.CatalogBlueprintImporter;
import com.day.cq.commons.jcr.JcrUtil;
import com.day.cq.wcm.api.Page;
import com.day.cq.wcm.api.PageManager;
import com.day.cq.wcm.api.WCMException;

/**
 * An abstract base class used for writing product importers.
 */
@Component(componentAbstract = true, metatype = true)
@Service
@ConsumerType
public abstract class AbstractBlueprintImporter extends AbstractImporter implements CatalogBlueprintImporter {
    private static final Logger log = LoggerFactory.getLogger(AbstractBlueprintImporter.class);

    protected String basePath = "/content/catalogs";

    protected long startTime;

    protected int catalogCount;                         // number of catalogs created
    protected int sectionCount;                         // number of catalog sections created

    public void importBlueprints(SlingHttpServletRequest request, SlingHttpServletResponse response) throws ServletException, IOException {

        if (!validateInput(request, response)) {
            return;
        }

        ResourceResolver resourceResolver = request.getResourceResolver();
        Session session = resourceResolver.adaptTo(Session.class);
        String storeName = request.getParameter("storeName");
        String storePath = request.getParameter("storePath");
        String provider = request.getParameter("provider");
        initTicker(request.getParameter("tickertoken"), session);

        Boolean incrementalImport = false;
        if (request.getParameter("incrementalImport") != null) {
            incrementalImport = true;
        }

        catalogCount = 0;
        sectionCount = 0;
        run(resourceResolver, storePath != null ? storePath : basePath, storeName, incrementalImport, provider);

        long millis = System.currentTimeMillis() - startTime;
        long seconds = millis / 1000;
        if (seconds > 120) {
            log.info("Imported " + catalogCount + " catalogs with " + sectionCount + " sections in " + seconds / 60 + " minutes.");
        } else {
            log.info("Imported " + catalogCount + " catalogs with " + sectionCount + " sections in " + seconds + " seconds.");
        }
        String msg = catalogCount + " catalogs created with " + sectionCount + " sections.";
        if (getErrorCount() > 0) {
            msg += " " + getErrorCount() + " errors encountered.";
        }
        if (catalogCount > 0) {
            respondWithMessages(response, msg);
        } else {
            response.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR, msg);
        }
    }

    /**
     * Validate any input required to run the importer.  Implementation to be supplied by concrete classes.
     * @param request
     * @param response
     * @return
     * @throws IOException
     */
    protected abstract boolean validateInput(SlingHttpServletRequest request, SlingHttpServletResponse response) throws IOException;

    /**
     * Creates the top-level catalog blueprint page.
     * @param pageManager
     * @param storePath
     * @param catalogTitle
     * @param session
     * @return
     * @throws RepositoryException
     * @throws WCMException
     */
    protected Page createCatalog(PageManager pageManager, String storePath, String catalogTitle, Session session)
            throws RepositoryException, WCMException {

        Page catalog = pageManager.create(storePath, mangleName(catalogTitle), "/libs/commerce/templates/catalog", catalogTitle);
        Node contentNode = catalog.getContentResource().adaptTo(Node.class);
        setProperty(contentNode, "templates/cq_rolloutConfigs", "[/etc/msm/rolloutconfigs/catalog]");
        catalogCount++;
        logMessage("Created catalog " + catalog.getPath(), false);
        checkpoint(session, false);
        return catalog;
    }

    /**
     * Creates a section blueprint within the catalog.
     * @param pageManager
     * @param parentPath
     * @param hierarchy
     * @param sectionTitle
     * @param session
     * @return
     * @throws RepositoryException
     * @throws WCMException
     */
    protected Page createSection(PageManager pageManager, String parentPath, String[] hierarchy, String sectionTitle, Session session)
            throws RepositoryException, WCMException {

        for (int i = 0; i < hierarchy.length - 1; i++) {
            parentPath += "/" + mangleName(hierarchy[i]);
        }
        String sectionName = hierarchy[hierarchy.length-1];
        Page section = pageManager.create(parentPath, mangleName(sectionName), "/libs/commerce/templates/section", sectionTitle);
        sectionCount++;
        logMessage("Created section " + section.getPath(), false);
        updateTicker(makeTickerMessage());
        checkpoint(session, false);
        return section;
    }

    /**
     * Sets a property on a catalog or section blueprint.  The path can be nested (ie: templates/product).
     * @param node
     * @param propertyPath
     * @param value
     * @throws RepositoryException
     */
    protected void setProperty(Node node, String propertyPath, String value) throws RepositoryException {
        String propertyName = propertyPath;
        if (propertyPath.contains("/")) {
            node = JcrUtil.createPath(node, propertyPath.substring(0, propertyPath.lastIndexOf("/")), false,
                                      "nt:unstructured", "nt:unstructured", node.getSession(), false);
            propertyName = propertyPath.substring(propertyPath.lastIndexOf("/") + 1);
        }
        if (value.startsWith("[") && value.endsWith("]")) {
            String[] values = value.substring(1, value.length()-1).split(",");
            node.setProperty(propertyName, values);
        } else {
            node.setProperty(propertyName, value);
        }
    }

    protected String makeTickerMessage() {
        return sectionCount + " section blueprints imported/updated";
    }


}
