/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2015 Adobe Systems Incorporated
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */

package com.adobe.cq.dam.cfm;

import aQute.bnd.annotation.ProviderType;
import org.apache.sling.api.adapter.Adaptable;

import java.util.Iterator;

/**
 * <p>Implementations of this interface allow accessing elements of a content fragment in a
 * way agnostic to the underlying data structures.</p>
 *
 * <p>Each content element consists of a set of 0..n variations, the element's content
 * being kind of a "master" variation all other variations derive from.</p>
 *
 * <p>A content element is defined to consist of textual content. The
 * {@link ContentFragment} interface is agnostic to a specific MIME type, so it's up to
 * the implementation to decide when and how to transform the text to a format that's
 * suitable for output.</p>
 *
 * <p>Transactional behavior: The caller is responsible for committing the respective
 * {@link org.apache.sling.api.resource.ResourceResolver} after calling one or more
 * methods that change a content element unless specified otherwise.</p>
 */
@ProviderType
public interface ContentElement extends Adaptable, Versionable {

    /**
     * Gets an iterator on the currently available variations of the element.
     *
     * @return iterator on variations
     */
    Iterator<ContentVariation> getVariations();

    /**
     * Gets the variation of the given name.
     *
     * @param variationName The name of the variation
     * @return The variation; <code>null</code> if no variation of the given name exists
     */
    ContentVariation getVariation(String variationName);

    /**
     * Creates a new variation of the element from the specified template.
     *
     * <p>The initial content of the new variation must be a copy of the current element
     * content.</p>
     *
     * <p>If the method is called multiple times with the same template, two different
     * variations have to created (with different names/identifiers).</p>
     *
     * <p>The specified template should be part of a {@link FragmentTemplate}.</p>
     *
     * @param template The template
     * @return The variation created
     * @throws ContentFragmentException if the variation could not be created
     */
    ContentVariation createVariation(VariationTemplate template)
            throws ContentFragmentException;

    /**
     * Removes the specified variation.
     *
     * <p>If the specified variation is not a member of the element, a
     * {@link IllegalArgumentException} has to be thrown.</p>
     *
     * @param variation The variation to remove
     * @throws ContentFragmentException if the variation could not be removed on the
     *                                  persistence level
     * @throws IllegalArgumentException if the specified variation is not part of this
     *                                  element
     */
    void removeVariation(ContentVariation variation) throws ContentFragmentException;

    /**
     * Gets the resolved variation for the given name.
     *
     * <p>The method must be able to handle names that do not refer to actually existing
     * variations. Instead, it has to handle this case in a well defined manner - usually
     * by falling back to another, actually existing variation or to the element's
     * default content.</p>
     *
     * <p>From a client's perspective, this method should be used to determine content
     * in any kind of "publishing context" - be it in the page authoring context or some
     * other context that prepares the final content to be presented to the user.</p>
     *
     * <p>How the resolution is eventually done is an implementation detail.</p>
     *
     * @param variationName The name of the variation; may be <code>null</code> (which
     *                      will return <code>null</code> to signal that default content
     *                      has to be used)
     * @return The resolved variation; <code>null</code> if there's no suitable variation
     *         and the element's default content has to be used
     */
    ContentVariation getResolvedVariation(String variationName);

    /**
     * <p>Gets the (technical) name of the content element.</p>
     *
     * @return The name of the content element
     */
    String getName();

    /**
     * Gets the (human-readable) title of the content element.
     *
     * @return The title of the content element
     */
    String getTitle();

    /**
     * <p>Gets the content of the content element.</p>
     *
     * @return The content of the content element
     */
    String getContent();

    /**
     * Gets the MIME type of the content element.
     *
     * @return The MIME type; refers to some text format
     */
    String getContentType();

    /**
     * <p>Sets the content of the content element.</p>
     *
     * <p>The specified content &amp; its MIME type must be textual. What MIME types are
     * eventually supported is left to the implementation.</p>
     *
     * @param content The new content of the content element
     * @throws ContentFragmentException if the content could not be written
     * @throws IllegalArgumentException if the specified MIME type is not supported or not
     *                                  a textual MIME type
     */
    void setContent(String content, String contentType) throws ContentFragmentException;

}
