/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2015 Adobe Systems Incorporated
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */

package com.adobe.cq.dam.cfm;

import aQute.bnd.annotation.ProviderType;
import org.apache.sling.api.resource.Resource;

import java.util.List;
import java.util.Map;

/**
 * Implementations of this interface provide means to manage content fragments.
 *
 * <p>Actions available from this interface:</p>
 *
 * <ul>
 *     <li>Creating new content fragments from a fragment template</li>
 *     <li>Utilities for handling associated content</li>
 * </ul>
 */
@ProviderType
public interface ContentFragmentManager {

    /**
     * Creates a new content fragment from the specified template.</p>
     *
     * <p>Transactional behavior: The caller is responsible for committing the respective
     * {@link org.apache.sling.api.resource.ResourceResolver} after the content fragment
     * was created by this method.</p>
     *
     * @param parent The parent for the new content fragment
     * @param template The template to be used
     * @param name The (technical) name of the fragment to be created
     * @param title The (human-readable) title of the fragment to be created
     * @return The newly created content fragment
     * @throws ContentFragmentException if the fragment could not be created
     */
    ContentFragment create(Resource parent, Resource template, String name, String title)
            throws ContentFragmentException;

    /**
     * Resolves the associated content of the specified content fragment.
     *
     * <p>This means that references to collections are resolved to their content, whereas
     * non-collections are simply taken over.</p>
     *
     * <p>The method returns a table, which contains a list of resolved resources (value)
     * per collection (key). Assets (like linked assets) are also added to this table, with
     * the asset being both the key and the only member of the value's list.</p>
     *
     * @param fragment The content fragment
     * @return List of resolved associated content, per collection/asset (for the latter,
     *         the asset is used as a key, and the value is a single list, containing the
     *         same asset)
     */
    Map<Resource, List<Resource>> resolveAssociatedContent(ContentFragment fragment);

    /**
     * Resolves the associated content of the specified content fragment.
     *
     * <p>This means that references to collections are resolved to their content, whereas
     * non-collections are simply taken over.</p>
     *
     * <p>This method returns a "flat" list of all resources that are either referenced
     * (collection) or part of the associated content themselves (asset).</p>
     *
     * <p>If a resource is part of multiple associated collection, it appears only once
     * in this list.</p>
     *
     * @param fragment The content fragment
     * @return List of resolved associated content, per collection/asset (for the latter,
     *         the asset is used as a key, and the value is a single list, containing the
     *         same asset)
     */
    List<Resource> resolveAssociatedContentFlat(ContentFragment fragment);

}
