/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2015 Adobe Systems Incorporated
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */

package com.adobe.cq.dam.cfm;

import aQute.bnd.annotation.ProviderType;

/**
 * <p>Implementations of this interface allow to access a variation of a content element
 * through a stable API, agnostic to the underlying content structure.</p>
 *
 * <p>Variations can be synchronized with some other element. If the implementation does
 * not provide enhances synchronization capabilities (which it is free to do), it should
 * synchronize with the element content.</p>
 *
 * <p>The synchronization works by using the synchronization source, taking their content,
 * optionally applying some rules and then updating the variation with that content.</p>
 *
 * <p>How the synchronization source is defined and how the source content is processed is
 * implementation specific. This API only provides means to manage the synchronization
 * state and triggering the synchronization.</p>
 *
 * <p>The synchronization state is represented by the {@link SyncStatus} enumeration.</p>
 *
 * <p>Transactional behavior: The caller is responsible for committing the respective
 * {@link org.apache.sling.api.resource.ResourceResolver} after calling one or more
 * methods that change a content fragment unless specified otherwise.</p>
 */
@ProviderType
public interface ContentVariation extends Versionable {

    /**
     * Gets the (technical) name of the variation.
     *
     * @return The name of the variation
     */
    String getName();

    /**
     * Gets the (human-readable) title of the variation.
     *
     * @return The title of the variation
     */
    String getTitle();

    /**
     * Gets a description for the variation.
     *
     * @return The description
     */
    String getDescription();

    /**
     * Gets the MIME type of the variation.
     *
     * @return The MIME type; refers to some text format
     */
    String getContentType();

    /**
     * Gets the content of the variation.
     *
     * @return The content of the variation
     */
    String getContent();

    /**
     * Sets the content of the variation.
     *
     * <p>The specified content &amp; its MIME type must be textual. What MIME types are
     * eventually supported is left to the implementation.</p>
     *
     * @param content The content
     * @param contentType The MIME type of the content
     * @throws ContentFragmentException if the content could not be written
     * @throws IllegalArgumentException if the specified MIME type is not supported or not
     *                                  a textual MIME type
     */
    void setContent(String content, String contentType) throws ContentFragmentException;

    /**
     * Gets the synchronization status of the variation.
     *
     * @return The synchronization state of the provided variation
     */
    SyncStatus getSyncStatus();

    /**
     * <p>Synchronizes the variation.</p>
     *
     * @throws ContentFragmentException if the synchronization failed
     */
    void synchronize() throws ContentFragmentException;

}
