/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2015 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.cq.dam.dm.process.api;

import aQute.bnd.annotation.ProviderType;
import com.day.cq.dam.api.Asset;
import com.day.cq.dam.api.Rendition;
import org.apache.sling.api.resource.ValueMap;

import java.io.File;
import java.io.IOException;

/**
 * Convert image content to PTIFF renditions.
 *
 * PTIFF rendition creation options:
 *
 * - usmAmount (double): Filter strength factor (0 - 5, defaults to 1.75)
 * - usmRadius (double): Filter kernel radius in pixels (0 - 250, defaults to 0.2)
 * - usmThreshold (int): Filter threshold level (0 - 255, defaults to 2)
 * - usmMonochrome (boolean): False will apply the filter to each color component separately, True will only apply it to the image brightness (defaults to false)
 * - dam.internal.OldPTiffRendition (boolean): True will write old style PTIFF renditions (defaults to false)
 * - xmpEmbed (boolean): True will embed the XMP data in the PTIFF rendition, False will not (default behavior).
 * - colorProfileEmbed (boolean) : If true, will embed the color profile (default behavior), false will not.
 * - maintainPixelType (boolean) : If true, will maintain the pixel information (default behavior), false will not.
 */
@ProviderType
public interface PTiffManager {

    /**
     * @return True if image processor is enabled
     */
    boolean isEnabled();

    /**
     * Find the PTIFF rendition for the given asset.
     *
     * @param asset Asset instance
     * @return PTIFF rendition or null if not available
     */
    PTiffRendition getPTiffRendition(Asset asset);

    /**
     * Convert an image rendition to a PTIFF rendition of the given asset.
     * Existing PTIFF rendition is not affected when createPTiffRendition returns null.
     *
     * @param rendition Image rendition
     * @param asset Asset where to write the PTIFF rendition and properties to
     * @param options PTIFF generation options
     * @return PTIFF rendition or null if the stream is not supported
     * @throws IOException when failing to access intermediary files, or the JCR
     */
    PTiffRendition createPTiffRendition(Rendition rendition, Asset asset, ValueMap options) throws IOException;

    /**
     * Convert an image file to a PTIFF rendition of the given asset.
     * Existing PTIFF rendition is not affected when createPTiffRendition returns null.
     *
     * @param file Image file
     * @param asset Asset where to write the PTIFF rendition and properties to
     * @param options PTIFF generation options
     * @return PTIFF rendition or null if the file is not supported
     * @throws IOException when failing to access the file, intermediary files, or the JCR
     */
    PTiffRendition createPTiffRendition(File file, Asset asset, ValueMap options) throws IOException;

    /**
     * Convert an image rendition to a PTIFF rendition of the given asset.
     * Existing PTIFF rendition is not affected when createPTiffRendition returns null.
     *
     * @param rendition Image rendition
     * @param asset Asset where to write the PTIFF rendition and properties to
     * @param options PTIFF generation options
     * @param inputFile File to write the rendition contents to
     * @param outputFile File to write the PTIFF result to
     * @return PTIFF rendition or null if the stream is not supported
     * @throws IOException when failing to access intermediary files, or the JCR
     */
    PTiffRendition createPTiffRendition(Rendition rendition, Asset asset, ValueMap options, File inputFile, File outputFile) throws IOException;

    /**
     * Remove PTIFF rendition from the asset
     *
     * @param asset Asset from which to remove the PTIFF rendition.
     */
    void removePTiffRendition(Asset asset);

}
