/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.cq.dam.index.bootstrap;

import com.adobe.cq.dam.index.builder.*;
import com.adobe.cq.dam.index.builder.api.IndexDefinition;
import com.adobe.cq.dam.index.helper.IndexHelper;
import com.google.common.collect.ImmutableList;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.ReferencePolicy;
import org.apache.jackrabbit.oak.plugins.index.IndexConstants;
import org.apache.sling.api.resource.LoginException;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceResolverFactory;
import org.osgi.service.component.ComponentContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;

/**
 * Dam indexing related bootstrap configuration
 */
@Component
public class DamIndexBootstrap {

    private static final Logger log = LoggerFactory.getLogger(DamIndexBootstrap.class);

    @Reference(policy = ReferencePolicy.STATIC)
    private ResourceResolverFactory jcrResolverFactory;

    private static String [] disableIndexList = {
            "dcFormat", "videoCodec", "damFileSize", "damSha1",
            "xmpMMDocumentId", "xmpMMInstanceId"
    };

    private static String [] depriortiseIndexList = {
            "cqTags",  "damStatus", "offTime", "onTime"
    };

    private static final String INDEX_ROOT_PATH = "/" + IndexConstants.INDEX_DEFINITIONS_NAME;

    protected void activate(final ComponentContext context) {
        try {
            ResourceResolver resolver = jcrResolverFactory.getAdministrativeResourceResolver(null);
            disableDefaultIndex(resolver);
            depriortiseIndex(resolver);
            createDefaultIndex(resolver);
        } catch(LoginException e) {
            log.error("Error while setting up dam index", e);
        }
    }

    private void disableDefaultIndex(ResourceResolver resolver) {
        for (String path : disableIndexList) {
            Resource indexRsc = resolver.getResource(INDEX_ROOT_PATH + "/" + path);
            if (indexRsc != null) {
                IndexHelper.disableIndex(indexRsc);
            }
        }
    }

    private void depriortiseIndex(ResourceResolver resolver) {
        for (String path : depriortiseIndexList) {
            Resource indexRsc = resolver.getResource(INDEX_ROOT_PATH + "/" + path);
            if (indexRsc != null) {
                IndexHelper.deprioritiseIndex(indexRsc);
            }
        }
    }

    private void createDefaultIndex(ResourceResolver resolver) {
        Resource indexRootRsc = resolver.getResource("/");
        IndexDefinition indexDef = new LuceneIndexDefinition("damAssetLucene",
                createAggregationRules(), createIndexRules());
        if (IndexHelper.getReIndexCount(
                indexRootRsc.getChild(IndexConstants.INDEX_DEFINITIONS_NAME + "/damAssetLucene")) <= 0) {
            if(indexDef.build(indexRootRsc, true)) {
                log.debug("IndexDefinition created successfully at {}", indexRootRsc.getPath());
            } else {
                log.debug("IndexDefinition creation failed");
            }
        }

        // create oak property index for property processingProfile - CQ-30652
        IndexDefinition oakPropertyIndexDefProcProf = new OakPropertyIndexDefinition(new String [] {"processingProfile"}, "processingProfile");
        if (IndexHelper.getReIndexCount(indexRootRsc.getChild(
                IndexConstants.INDEX_DEFINITIONS_NAME + "/processingProfile")) <= 0) {
            oakPropertyIndexDefProcProf.build(indexRootRsc, true);
        }

        // create oak property index for property processingProfile - CQ-30652
        IndexDefinition oakPropertyIndexDefRefExp = new OakPropertyIndexDefinition(new String [] {"refExpired"}, "refExpired");
        if (IndexHelper.getReIndexCount(indexRootRsc.getChild(
                IndexConstants.INDEX_DEFINITIONS_NAME + "/refExpired")) <= 0) {
            oakPropertyIndexDefRefExp.build(indexRootRsc, true);
        }
        // CQ-37210
        IndexDefinition oakPropertyIndexDefSubType = new OakPropertyIndexDefinition(new String [] {"subType"}, "subType");
        if (IndexHelper.getReIndexCount(indexRootRsc.getChild(
                IndexConstants.INDEX_DEFINITIONS_NAME + "/subType")) <= 0) {
            oakPropertyIndexDefSubType.build(indexRootRsc, true);
        }

        IndexDefinition profileIndexDef = new OakPropertyIndexDefinition(new String [] {"metadataProfile", "videoProfile", "imageProfile"}, "profiles");
        if (IndexHelper.getReIndexCount(indexRootRsc.getChild(
                IndexConstants.INDEX_DEFINITIONS_NAME + "/profiles")) <= 0) {
            profileIndexDef.build(indexRootRsc, true);
        }
    }

    public static List<AggregationRule> createAggregationRules() {
        return ImmutableList.of(
                new AggregationRule("dam:Asset",
                        ImmutableList.of("jcr:content", "jcr:content/metadata", "jcr:content/metadata/*", "jcr:content/renditions",
                                "jcr:content/renditions/original", "jcr:content/renditions/original/jcr:content", "jcr:content/comments", "jcr:content/comments/*", "jcr:content/usages")
                )
        );
    }

    public static List<IndexRule> createIndexRules() {
        return ImmutableList.of(
                new IndexRule("dam:Asset",
                        ImmutableList.of(
                                new SimplePropertyDefinition("jcr:content/metadata/cq:tags", "cqTags", 1.0f, false, true, true),
                                new SimplePropertyDefinition("jcr:content/metadata/dc:format", "dcFormat", 1.0f, true),
                                new SimplePropertyDefinition("jcr:content/metadata/dam:status", "damStatus"),
                                new SimplePropertyDefinition("jcr:content/metadata/videoCodec", "videoCodec"),
                                new SimplePropertyDefinition("jcr:content/metadata/audioCodec", "audioCodec"),
                                new SimplePropertyDefinition("jcr:content/metadata/dc:title", "dcTitle", 2.0f, false, true, true),
                                new SimplePropertyDefinition("jcr:content/metadata/dc:description", "dcDescription", 1.0f, false, true, true),
                                new SimplePropertyDefinition("jcr:content/metadata/xmpMM:InstanceID", "xmpMMInstanceId"),
                                new SimplePropertyDefinition("jcr:content/metadata/xmpMM:DocumentID", "xmpMMDocumentId"),
                                new SimplePropertyDefinition("jcr:content/metadata/dam:sha1", "damSha1"),
                                new SimplePropertyDefinition("jcr:content/hasValidMetadata", "hasValidMetadata", "Boolean"),
                                new SimplePropertyDefinition("jcr:content/metadata/videoBitrate", "videoBitrate"),
                                new SimplePropertyDefinition("jcr:content/metadata/audioBitrate", "audioBitRate"),
                                new SimplePropertyDefinition("jcr:content/usages/usedBy", "usedBy"),
                                new OrderedPropertyDefinition("jcr:content/jcr:lastModified", "jcrLastModified", "Date"),
                                new OrderedPropertyDefinition("jcr:content/metadata/prism:expirationDate", "expirationDate", "Date"),
                                new OrderedPropertyDefinition("jcr:content/onTime", "onTime", "Date"),
                                new OrderedPropertyDefinition("jcr:content/offTime", "offTime", "Date"),
                                new OrderedPropertyDefinition("jcr:content/metadata/dam:size", "damSize", "Long"),
                                new OrderedPropertyDefinition("jcr:content/averageRating", "averageRating", "Double"),
                                new OrderedPropertyDefinition("jcr:content/metadata/tiff:ImageWidth", "tiffImageWidth", "Long"),
                                new OrderedPropertyDefinition("jcr:content/metadata/tiff:ImageLength", "tiffImageLength", "Long"),
                                new OrderedPropertyDefinition("jcr:content/usages/dam:score", "score", "Long"),
                                new NullChkEnabledPropertyDefinition("jcr:content/metadata/simsearchVersion", "simsearchVersion", true),
                                new SimplePropertyDefinition("jcr:content/metadata/predictedTags/*/name", "predictedTags", true, false, true)
                        )
                )
        );
    }
}
