/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 ************************************************************************/
package com.adobe.cq.dam.index.builder;

import com.adobe.cq.dam.index.builder.api.IndexDefinition;
import com.google.common.collect.ImmutableMap;
import org.apache.jackrabbit.JcrConstants;
import org.apache.jackrabbit.oak.plugins.index.IndexConstants;
import org.apache.jackrabbit.oak.plugins.index.lucene.IndexFormatVersion;
import org.apache.jackrabbit.oak.plugins.index.lucene.LuceneIndexConstants;
import org.apache.sling.api.resource.PersistenceException;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;

/**
 * Represents Oak Lucene Index Definition
 */
public class LuceneIndexDefinition implements IndexDefinition {

    private static Logger log = LoggerFactory.getLogger(LuceneIndexDefinition.class);

    String indexName;
    List<IndexRule> indexRules;
    List<AggregationRule> aggregationRules;

    /**
     * Initializes LuceneIndexDefinition with given aggregation and index rules
     * @param indexName indexName with which the index will be saved in repository
     * @param aggregationRules list of AggregationRule
     * @param indexRules list of IndexRule
     */
    public LuceneIndexDefinition(String indexName,
                                 List<AggregationRule> aggregationRules, List<IndexRule> indexRules) {
        this.indexName = indexName;
        this.aggregationRules = aggregationRules;
        this.indexRules = indexRules;
    }

    /**
     * {@inheritDoc}
     */
    public boolean build(Resource resource) {
        return build(resource, false);
    }

    /**
     * {@inheritDoc}
     */
    public boolean build(Resource resource, boolean reindex) {
        if (resource == null) {
            log.warn("LuceneIndexDefinition build failed due to null resource");
            return false;
        }
        ResourceResolver resolver = resource.getResourceResolver();
        try {
            // get or create oak:index node
            Resource oakIndexNode = resolver.getResource(resource, IndexConstants.INDEX_DEFINITIONS_NAME);
            if (oakIndexNode == null) {
                oakIndexNode = resolver.create(resource, IndexConstants.INDEX_DEFINITIONS_NAME,new ImmutableMap.Builder<String, Object>()
                                .put(JcrConstants.JCR_PRIMARYTYPE, "nt:unstructured").build()
                );
            }
            log.info(oakIndexNode.getPath());
            // if index definition exists, delete and recreate
            if (resolver.getResource(oakIndexNode, indexName) != null) {
                resolver.delete(resolver.getResource(oakIndexNode, indexName));
            }

            // create IndexDefinition node
            Resource indexDefNode = resolver.create(oakIndexNode, indexName, new ImmutableMap.Builder<String, Object>()
                            .put(JcrConstants.JCR_PRIMARYTYPE, IndexConstants.INDEX_DEFINITIONS_NODE_TYPE)
                            .put(IndexConstants.ASYNC_PROPERTY_NAME, "async")
                            .put(LuceneIndexConstants.COMPAT_MODE, IndexFormatVersion.V2.getVersion())
                            .put(IndexConstants.TYPE_PROPERTY_NAME, LuceneIndexConstants.TYPE_LUCENE)
                            .put(LuceneIndexConstants.EVALUATE_PATH_RESTRICTION, true)
                            .put(IndexConstants.REINDEX_PROPERTY_NAME, reindex).build()
            );

            // create Aggregates node
            Resource aggregatesNode = resolver.create(indexDefNode,
                    LuceneIndexConstants.AGGREGATES, ImmutableMap.of(JcrConstants.JCR_PRIMARYTYPE, (Object)"nt:unstructured"));

            for (AggregationRule aggRule : aggregationRules) {
                aggRule.build(aggregatesNode);
            }

            // create IndexRules node
            Resource indexRulesNode = resolver.create(indexDefNode,
                    LuceneIndexConstants.INDEX_RULES, ImmutableMap.of(JcrConstants.JCR_PRIMARYTYPE, (Object)"nt:unstructured"));

            for (IndexRule indexRule : indexRules) {
                indexRule.build(indexRulesNode);
            }

            resolver.commit();
            return true;
        } catch (PersistenceException e) {
            log.error("Error while creating LuceneIndexDefinition at {}", resource.getPath(), e);
        }
        return false;
    }
}
