/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 ************************************************************************/
package com.adobe.cq.dam.index.builder;

import com.adobe.cq.dam.index.builder.api.IndexDefinition;
import com.google.common.collect.ImmutableMap;
import org.apache.jackrabbit.JcrConstants;
import org.apache.jackrabbit.oak.plugins.index.IndexConstants;
import org.apache.sling.api.resource.PersistenceException;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.jcr.Node;
import javax.jcr.PropertyType;
import javax.jcr.RepositoryException;


/**
 * Implements PropertyDefinition of Oak Property Index
 */
public class OakPropertyIndexDefinition implements IndexDefinition {

    String nodeName;
    String [] properties;
    private static Logger log = LoggerFactory.getLogger(OakPropertyIndexDefinition.class);

    /**
     * Creates a Oak PropertyIndex Definition
     * @param property property to index
     * @param nodeName PropertyDefinition node name
     */
    public OakPropertyIndexDefinition(String [] property, String nodeName) {
        setNodeName(nodeName);
        setProperty(property);
    }

    public void setNodeName(String nodeName) {
        this.nodeName = nodeName;
    }

    public void setProperty(String [] prop) {
        this.properties = prop;
    }

    /**
     * {@inheritDoc}
     */
    public boolean build(Resource resource) {
        return build(resource, false);
    }

    /**
     * {@inheritDoc}
     */
    public boolean build(Resource resource, boolean reindex) {
        if (resource == null) {
            log.warn("OakPropertyIndexDefinition build failed due to null resource");
            return false;
        }
        ResourceResolver resolver = resource.getResourceResolver();
        try {
            // get or create oak:index node
            Resource oakIndexNode = resolver.getResource(resource, IndexConstants.INDEX_DEFINITIONS_NAME);
            if (oakIndexNode == null) {
                oakIndexNode = resolver.create(resource, IndexConstants.INDEX_DEFINITIONS_NAME,new ImmutableMap.Builder<String, Object>()
                                .put(JcrConstants.JCR_PRIMARYTYPE, "nt:unstructured").build()
                );
            }
            log.info(oakIndexNode.getPath());
            // if index definition exists, delete and recreate
            if (resolver.getResource(oakIndexNode, nodeName) != null) {
                resolver.delete(resolver.getResource(oakIndexNode, nodeName));
            }

            // create IndexDefinition node
            Resource indexDef = resolver.create(oakIndexNode, nodeName, new ImmutableMap.Builder<String, Object>()
                            .put(JcrConstants.JCR_PRIMARYTYPE, IndexConstants.INDEX_DEFINITIONS_NODE_TYPE)
                            .put(IndexConstants.TYPE_PROPERTY_NAME, "property")
                            .put(IndexConstants.REINDEX_PROPERTY_NAME, reindex).build()
            );
            Node indexDefNode = indexDef.adaptTo(Node.class);
            indexDefNode.setProperty(IndexConstants.PROPERTY_NAMES, properties, PropertyType.NAME);
            resolver.commit();
            return true;
        } catch (PersistenceException e) {
            log.error("Error while creating OakPropertyIndexDefinition at {}", resource.getPath(), e);
        } catch (RepositoryException e) {
            log.error("Error while creating OakPropertyIndexDefinition at {}", resource.getPath(), e);
        }
        return false;
    }
}
