/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.cq.mcm.salesforce;

import com.adobe.granite.crypto.CryptoException;
import com.adobe.granite.crypto.CryptoSupport;
import com.day.cq.commons.jcr.JcrConstants;
import com.day.cq.wcm.webservicesupport.Configuration;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.Service;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.commons.json.JSONException;
import org.apache.sling.commons.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

/**
 * The <code>SalesforceLeadSearch</code> component is used to search leads on salesforce and return back the data
 * as JSON
 * */
@Component
@Service(value = SalesforceLeadSearch.class)
public class SalesforceLeadSearch {

    private static final Logger log = LoggerFactory.getLogger(SalesforceLeadSearch.class);
    private static final String INSTANCE_URL = "instanceurl";
    public static final String CUSTOMER_KEY = "customerkey";
    public static final String CUSTOMER_SECRET = "customersecret";
    public static final String REFRESH_TOKEN = "refreshtoken";
    private static final String QUERY_PATH = "/services/data/v20.0/query/";
    public static final String APPLICATION_JSON = "application/json";
    public static final String ACCESS_TOKEN = "accesstoken";

    @Reference
    private CryptoSupport cryptoSupport;

    /**
     * Uses the <code>{@link SalesforceClient}</code> to search leads
     * @param cloudConfig Salesforce Cloud Service Configuration used for identifying Salesforce Account.
     * @param parameters Search Parameters used via searching
     * @param resolver The current ResourceResolver object used if the access token expires during the search
     *                 and we need to save the new access token in the configuration.
     * @return JSON object consisting of leads information
     * */
    public JSONObject search(Configuration cloudConfig, SalesforceSearchParameters parameters,
                             ResourceResolver resolver)throws SalesforceException{

        if (cloudConfig!=null){
            String instanceUrl  = cloudConfig.get(INSTANCE_URL, "");
            String accessToken  = cloudConfig.get(ACCESS_TOKEN, "");
            String clientId     = cloudConfig.get(CUSTOMER_KEY, "");
            String encryptedCustomerSecret = cloudConfig.get(CUSTOMER_SECRET, "");
            String encryptedRefereshToken = cloudConfig.get(REFRESH_TOKEN, "");

            try {
                String customerSecret = encryptedCustomerSecret;
                String refreshToken = encryptedRefereshToken;
                if(cryptoSupport.isProtected(encryptedCustomerSecret)){
                    customerSecret = cryptoSupport.unprotect(encryptedCustomerSecret);
                }
                if(cryptoSupport.isProtected(encryptedRefereshToken)){
                    refreshToken = cryptoSupport.unprotect(encryptedRefereshToken);
                }

                SalesforceClient client = new SalesforceClient();
                client.setAccessToken(accessToken);
                client.setInstanceURL(instanceUrl);
                client.setRefreshToken(refreshToken);
                client.setClientId(clientId);
                client.setClientSecret(customerSecret);
                client.setMethod(SalesforceClient.AvailableMethods.GET);
                client.setContentType(APPLICATION_JSON);
                client.setPath(QUERY_PATH);
                client.addParameter("q", buildSOSL(parameters));
                SalesforceResponse response = client.executeRequest();

                if(response.getAccessTokenUpdated()){
                    // Update Access token in Configuration
                    String configPath = cloudConfig.getPath();
                    Resource configResource = resolver.getResource(configPath);
                    Node configNode = configResource.adaptTo(Node.class).getNode(JcrConstants.JCR_CONTENT);
                    configNode.setProperty(ACCESS_TOKEN, client.getAccessToken());
                    configNode.getSession().save();
                }
                return response.getBodyAsJSON();

            } catch (RepositoryException e) {
                log.error("Repository Exception in Searching SFDC Leads "+e.getMessage());
                throw new SalesforceException("Repository Exception in Searching SFDC Leads "+e.getMessage());
            } catch (CryptoException e) {
                log.error("Cryto Exception in searching SFDC Leads " + e.getMessage());
                throw new SalesforceException("Crypto Exception in searching SFDC Leads "+e.getMessage());
            } catch (JSONException e) {
                log.error("JSON Exception in searching SFDC Leads "+e.getMessage());
                throw new SalesforceException("JSON Exception in searching SFDC Leads "+e.getMessage());
            }
        }

        return null;
    }


    /***
     * Builds a SOSL query from the given <code>{@link SalesforceSearchParameters}</code> parameters
     * */
    protected String buildSOSL(SalesforceSearchParameters parameters) throws SalesforceException{

        StringBuilder query = new StringBuilder();
        query.append("SELECT ");

        if(parameters.getResultProperties()!=null && parameters.getResultProperties().length>0){
            for(int i=0; i<parameters.getResultProperties().length; i++){
                if(parameters.getResultProperties()[i]!=null){
                    query.append(parameters.getResultProperties()[i] + ", ");
                }
            }
            // Remove the last ,
            query.deleteCharAt(query.lastIndexOf(","));
        }
        else{
            query.append("FirstName, LastName, Company, Status, Email ");
        }

        if(SalesforceSearchParameters.SalesforceObjectType.LEAD.equals(parameters.getObjectType())){
            query.append("FROM LEAD ");
        }
        else if(SalesforceSearchParameters.SalesforceObjectType.CONTACT.equals(parameters.getObjectType())){
            query.append("FROM CONTACT ");
        }
        else {
            throw new SalesforceException("Invalid Search Query: Must search for either Leads or Contacts");
        }

        if(parameters.getSearchOperator()!=null && parameters.getSearchType()!=null &&
                parameters.getSearchVal()!=null){

            query.append("WHERE "+ parameters.getSearchType()+" "+parameters.getSearchOperator()+" "
                    +getEncodedSearchVal(parameters.getSearchOperator(), parameters.getSearchVal()));
        }

        return query.toString();
    }

    private String getEncodedSearchVal(String operator, String searchVal){
        // If its a number don't surround within quotes
        Double searchValue = null;
        try {
            searchValue = Double.parseDouble(searchVal);
            return searchValue.toString();
        } catch (NumberFormatException e) {
            return "'"+searchVal+"'";
        }
    }

}
