/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.mobile.angular.data.util;

import com.adobe.cq.mobile.platform.MobileResource;
import com.adobe.cq.mobile.platform.MobileResourceLocator;
import com.adobe.cq.mobile.platform.MobileResourceType;
import com.day.cq.wcm.api.NameConstants;
import com.day.cq.wcm.api.Page;
import com.day.cq.wcm.api.PageFilter;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ValueMap;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * Utility class for exporting content in a manner consistent with client side framework
 * expectations
 *
 * @since CQ 6.0
 *
 * AdobePatentID="3773US01"
 */
public class FrameworkContentExporterUtils {

    // Resource type that all Angular pages should extend
    public static final String NG_PAGE_RESOURCE_TYPE = "mobileapps/components/angular/ng-page";
    public static final String NG_FRAMEWORK_TYPE_ID = "angular";

    /**
     * jcr:content node name
     */
    private static final String JCR_CONTENT = "jcr:content/";

    /**
     * Returns a relative path from the parentResource to the descendantResource.
     *
     * @param parentResource     resulting path will be relative from this resource
     * @param descendantResource descendant resource. Must be a descendant of parentResource
     * @return a relative path or null if the descendantResource is not a descendant of parentResource
     */
    public static String getRelativePathToDescendantResource(Resource parentResource, Resource descendantResource) {
        return getRelativePathToDescendantPath(parentResource, descendantResource.getPath());
    }

    /**
     * Returns a relative path from the parentResource to the absoluteDescendantPath
     *
     * @param parentResource         resulting path will be relative from this resource
     * @param absoluteDescendantPath path to a descendant of parentResource
     * @return a relative path or null if absoluteDescendantPath does not
     *         represent a descendant of parentResource
     */
    public static String getRelativePathToDescendantPath(Resource parentResource, String absoluteDescendantPath) {
        String parentPath = parentResource.getPath();

        // By definition, child path must start with the parent path
        if (absoluteDescendantPath.indexOf(parentPath) == 0) {
            String relativeResourcePath = parentResource.getName() + absoluteDescendantPath.substring(parentPath.length());

            return relativeResourcePath;
        }
        return null;
    }

    /**
     * Returns a path to the given asset.
     *
     * @param parentResource    path will be relative to this resource if appExport is true
     * @param absoluteDescendantAssetPath   absolute path to a descendant of parentResource
     * @param appExport if true, content is being exported
     * @return an absolute path if appExport is false, relative path (to parentResource) if appExport is true
     */
    public static String getPathToAsset(Resource parentResource, String absoluteDescendantAssetPath, boolean appExport) {
        // If this page is not being exported, return the absolute path
        if (appExport == false) {
            return absoluteDescendantAssetPath;
        }
        // else: we're exporting this content. Prepare the path accordingly
        String relativeResourcePath = getRelativePathToDescendantPath(parentResource, absoluteDescendantAssetPath);
        relativeResourcePath = replaceJcrContent(relativeResourcePath);
        return relativeResourcePath;
    }

    /**
     * Returns a resource that is determined to be the top-level page in an application that resource
     * is a descendant of.
     *
     * @param resource the resource in question
     * @return an ancestor of resource, or resource itself if it is the top-level page
     */
    public static Resource getTopLevelAppResource(Resource resource) {
        Resource parent = resource.getParent();
        // If we're at the root
        // OR the parent is not a page
        // OR the parent is an app instance or content page
        // THEN we're at the top
        if (parent == null || (!NameConstants.NT_PAGE.equals(parent.getResourceType()))) {
            return resource;
        } else {
            MobileResource parentMobileRes = parent.adaptTo(MobileResource.class);
            if (parentMobileRes.isA(MobileResourceType.CONTENT.getType(), MobileResourceType.INSTANCE.getType())) {
                return resource;
            }
        }
        // else; keep looking
        return getTopLevelAppResource(parent);
    }

    /**
     * Determines if this resource is the top level app resource.
     *
     * @param resource the resource in question
     * @return true if this page is the top level app resource
     */
    public static boolean isTopLevelAppResource(Resource resource) {
        if (resource == null) {
            return false;
        }
        return resource.equals(getTopLevelAppResource(resource));
    }

    /**
     * Returns the number of milliseconds since the UNIX epoch.
     * @return long, the current unix time
     */
    public static long getMillisecondsSinceUnixEpoch() {
        return System.currentTimeMillis();
    }

    public static List<Page> getAllAngularDescendantPages(Page page, PageFilter pageFilter) {
        List<Page> pageList = new ArrayList<Page>();
        getAllDescendantsOfResourceType(page, FrameworkContentExporterUtils.NG_PAGE_RESOURCE_TYPE, pageFilter, pageList);
        return pageList;
    }

    /**
     * Returns a Javascript-friendly String to identify this resource.
     * @param resourcePath absolute path to the resource
     * @return String with all alphanumeric characters removed and trimmed to the relative
     * component path if jcr:content/ is found
     */
    public static String getJsFriendlyResourceName(String resourcePath) {
        if (resourcePath == null) {
            return null;
        }

        // Trim to substring following jcr:content/ if found
        String uniqueName = resourcePath;
        String key = "jcr:content/";
        int indexOfKey = uniqueName.indexOf(key);
        if (indexOfKey > 0) {
            uniqueName = uniqueName.substring(indexOfKey + key.length());
        }
        // if the key is not found: we're dealing with a page rather than a component so the
        // name must be unique against all other pages in the app, so we use the whole path

        // Remove all non-alphanumeric characters
        return uniqueName.replaceAll("[^A-Za-z0-9]", "");
    }

    /**
     * Returns a relative path to the root level nodes in the repository.
     *
     * @param resource the resource from which to derive the path
     * @return relative path to content located at the root of the repository
     */
    public static String getRelativePathToRootLevel(Resource resource) {
        String path = "";
        Resource ancestor = resource.getParent();
        while ((ancestor = ancestor.getParent()) != null) {
            path = path + "../";
        }
        return path;
    }

    /**
     * Returns a list of all the 'angular' components that are descendants of
     * the given resource.
     *
     * @param pageContentResource resource to search for components
     * @return list of 'angular' components
     */
    public static List<Resource> getAllAngularPageComponents(Resource pageContentResource) {
        List<Resource> angularComponents = new ArrayList<Resource>();
        getAllAngularPageComponentsHelper(pageContentResource, angularComponents);

        return angularComponents;
    }

    /**
     * Determines the nearest ancestor template resource. This enables multiple similar
     * resources to share the same template.
     *
     * @param resource A resource that requires a template to render
     * @return template for the given resource, or null if this resource has no
     * template resource
     */
    public static Resource getAncestorTemplateResource(Resource resource, String resourceType) {
        if (resource == null || resourceType == null) {
            return null;
        }

        Resource contentResource = resource.getChild("jcr:content");
        if (contentResource != null && contentResource.isResourceType(resourceType)) {
            return resource;
        }

        // else
        return getAncestorTemplateResource(resource.getParent(), resourceType);
    }

    /**
     * Trims a component path down to the substring following "jcr:content/".
     *
     * @param path
     * @return substring of the path following "jcr:content"
     */
    public static String getRelativeComponentPath(String path) {
        if (path == null || !path.contains(JCR_CONTENT)) {
            return path;
        }

        return path.substring(path.indexOf(JCR_CONTENT) + JCR_CONTENT.length());
    }

    private static void getAllAngularPageComponentsHelper(Resource resource, List<Resource> components) {
        Iterator<Resource> resourceChildrenIter = resource.listChildren();

        while (resourceChildrenIter.hasNext()) {
            Resource childResource = resourceChildrenIter.next();
            ValueMap childProperties = childResource.adaptTo(ValueMap.class);
            if (NG_FRAMEWORK_TYPE_ID.equals(childProperties.get("frameworkType", String.class))) {
                components.add(childResource);
            }
            getAllAngularPageComponentsHelper(childResource, components);
        }
    }

    private static void getAllDescendantsOfResourceType(Page page, String resourceType, PageFilter pageFilter, List<Page> list) {
        Iterator<Page> children = page.listChildren(pageFilter, true);
        while (children.hasNext()) {
            Page child = children.next();
            Resource contentResource = child.getContentResource();
            // Only add if it's the given resourceType
            if (contentResource != null &&
                    contentResource.isResourceType(resourceType)) {
                list.add(child);
            }
        }
    }

    /*
     * When content is exported to the file system, jcr:content must be converted to a valid
     * directory name. _jcr_content was not used because Android devices exclude directories
     * beginning with '_' at app compile time.
     */
    private static String replaceJcrContent(String path) {
        if (path == null) {
            return null;
        }
        return path.replaceAll("/jcr:content/", "/jcr_content/");
    }
}
