/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.mobile.dps;

import com.day.cq.wcm.api.NameConstants;
import com.day.cq.wcm.api.Page;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceUtil;
import org.apache.sling.api.resource.ValueMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.jcr.RepositoryException;

/**
 * Media utilities
 */
public class MediaUtil {

    /**
     * Static logger
     */
    private static final Logger LOGGER = LoggerFactory.getLogger(MediaUtil.class);

    /**
     * Default constructor
     */
    private MediaUtil() {
        // no instance
    }

    /**
     * Get the image selector to render the given image resource
     *
     * @param imageResource image resource being rendered
     * @return image selector string
     */
    public static String getMediaImageSelector(Resource imageResource) throws RepositoryException {
        String selector = MediaConstants.DEFAULT_IMAGE_SELECTOR;
        if(isOriginalImageRequested(imageResource)){
            selector = MediaConstants.DEFAULT_IMAGE_SELECTOR + ".original";
        } else if (isImageScalingEnabled(imageResource)) {
            String adaptiveSelector = MediaConstants.DEFAULT_IMAGE_SELECTOR;
            String adaptiveWidth = String.valueOf(getImageWidthSelector(imageResource));
            String adaptiveQuality = getImageQualitySelector(imageResource);
            selector = adaptiveSelector + "." + adaptiveWidth + "." + adaptiveQuality;
        } else {
            // Use default image
            LOGGER.info("Media: Scaling disabled. Image will not be scaled, default rendering will be used");
        }
        return selector;
    }
    
    /**
     * Should the image be scaled.
     *
     * @param imageResource image resource being rendered
     * @return true if the image should be scaled an false otherwise
     */
    private static boolean isOriginalImageRequested(Resource imageResource) {
        return isBooleanPropertyTrue(imageResource, MediaConstants.PN_IS_ORIGINAL_IMAGE_REQUESTED);
    }

    /**
     * Should the image be scaled.
     *
     * @param imageResource image resource being rendered
     * @return true if the image should be scaled an false otherwise
     */
    private static boolean isImageScalingEnabled(Resource imageResource) {
        return isBooleanPropertyTrue(imageResource, MediaConstants.PN_IS_SCALING_ENABLED);
    }

    /**
     * Is a boolean property set to true
     *
     * @param imageResource image resource being rendered
     * @param propertyName the boolean property name
     * @return true if set to true, false otherwise
     */
    private static boolean isBooleanPropertyTrue(Resource imageResource, String propertyName) {
        ValueMap properties = ResourceUtil.getValueMap(imageResource);
        boolean isBooleanPropertyTrue = false;
        if (properties.containsKey(propertyName)) {
            isBooleanPropertyTrue = properties.get(propertyName, Boolean.class);
        }
        return isBooleanPropertyTrue;
    }

    /**
     * Get the width selector to use.
     *
     * @param imageResource image resource being rendered
     * @return the width selector to use
     */
    private static String getImageWidthSelector(Resource imageResource) {
        ValueMap properties = ResourceUtil.getValueMap(imageResource);

        // Default to full
        String targetWidth = MediaConstants.IMAGE_WIDTH_SELECTOR_FULL;

        // Check overrides
        String setWidthOverride = properties.get(MediaConstants.PN_IMAGE_SCALE_WIDTH, String.class);
        if (setWidthOverride != null) { // Resource override
            targetWidth = setWidthOverride;
        }
        return targetWidth;
    }
    
    /**
     * Get image quality selector.
     *
     * @param imageResource image resource being rendered
     * @return image quality selector
     */
    private static String getImageQualitySelector(Resource imageResource) {
        ValueMap properties = ResourceUtil.getValueMap(imageResource);
        return properties.get(MediaConstants.PN_IMAGE_QUALITY, MediaConstants.IMAGE_QUALITY_SELECTOR_FULL);
    }

    /**
     * Get the parent page this resource belongs to.
     *
     * @param resource the resource in question
     * @return get the parent page this resource belongs to, null if not found
     */
    private static Page getParentPage(Resource resource) {
        Resource parentResource = resource.getParent();
        if (parentResource != null) {
            if (ResourceUtil.isA(parentResource, NameConstants.NT_PAGE)) {
                return parentResource.adaptTo(Page.class);
            } else {
                return getParentPage(parentResource);
            }
        }
        return null;
    }
}
