/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.mobile.dps;

import java.util.HashMap;
import java.util.Map;

/**
 * Enum of DPS ProtectedAccess
 *
 * @since CQ5.6
 */
public enum ProtectedAccess {
    /**
     * Article cannot be shared. Closed maps to “Protected” in the Folio Producer Editor setting.
     */
    PROTECTED("protected"),

    /**
     * @deprecated Use PROTECTED instead.
     */
    @Deprecated
    CLOSED("Closed"),

    /**
     * Article can be shared after publishing via social sharing, subject to paywall threshold rules on both touch
     * devices and in the web viewer. Open maps to “Metered” in the Folio Producer Editor setting.
     */
    METERED("metered"),

    /**
     * @deprecated Use METERED instead.
     */
    @Deprecated
    OPEN("Open"),

    /**
     * Article can be freely shared on touch devices but are subject to paywall threshold rules for the web viewer (new
     * in R26) (only available for articles that target viewer 26 or greater)
     */
    FREE("free");

    /**
     * Other supported values
     */
    private static final String METERED_OPEN = "Open"; // Classic
    private static final String PROTECTED_CLOSED = "Closed"; // Classic

    /**
     * Enum map for lookup
     */
    private static Map<String, ProtectedAccess> ENUM_BY_VALUE;

    /**
     * Used to map the property name to the enum value
     */
    static {
        ENUM_BY_VALUE = new HashMap<String, ProtectedAccess>(ProtectedAccess.values().length);
        for (ProtectedAccess enumValue : ProtectedAccess.values()) {
            if(enumValue.equals(CLOSED) || enumValue.equals(OPEN)){
                // IGNORE DEPRECATED ENUMS
            } else {
                ENUM_BY_VALUE.put(enumValue.getValue().toLowerCase(), enumValue); //forgiving, use toLowerCase
            }
        }
    }

    /**
     * The value for this enum
     */
    private String value;

    /**
     * Create a ProtectedAccess
     *
     * @param value The property value
     */
    private ProtectedAccess(String value) {
        this.value = value;
    }

    /**
     * @return ProtectedAccess value
     */
    public String getValue() {
        return value;
    }

    /**
     * Returns true if this is a ProtectedAccess and false otherwise.
     *
     * @param value The value to check
     * @return true if a ProtectedAccess and false otherwise
     */
    public static final boolean isProtectedAccess(String value) {
        return getProtectedAccess(value) != null;
    }

    /**
     * Return the ProtectedAccess object for the specified value
     *
     * @param value used to specify the ProtectedAccess
     * @return a ProtectedAccess for the specified value or null if invalid
     */
    public static final ProtectedAccess getProtectedAccess(String value) {
        ProtectedAccess protectedAccess = null;
        if(value!=null){
            value = value.toLowerCase();
            protectedAccess = ENUM_BY_VALUE.get(value);
            if(protectedAccess==null){ // Try old values
                if(value.equals(METERED_OPEN.toLowerCase())){
                    protectedAccess = METERED;
                } else if(value.equals(PROTECTED_CLOSED.toLowerCase())){
                    protectedAccess = PROTECTED;
                }
            }
        }
        return protectedAccess;
    }
}
