package com.adobe.cq.mobile.platform;

import aQute.bnd.annotation.ProviderType;
import org.apache.sling.api.resource.Resource;

/**
 * The <code>MobileResourceLocator</code> provides methods to assist with finding mobile applications.
 */
@ProviderType
public interface MobileResourceLocator {
    /**
     * Find all child resources matching the requested app type.
     * @param resource {@link Resource } The resource to start searching from.
     * @param type Mobile resource type id. Use {@link com.adobe.cq.mobile.platform.MobileResourceType} for common app types.
     * @return A list of all child resources matching the requested type.
     */
    Iterable<MobileResource> findResourcesByType(Resource resource, String type);

    /**
     * Traverse over child resources to locate a resource matching the requested app type.
     * @param resource {@link Resource } The resource to start iterating from
     * @param type Mobile resource type id. Use {@link com.adobe.cq.mobile.platform.MobileResourceType} for common app types.
     * @param deep If <code>true</code> the child resources will be also analyzed
     *
     * @return The first resource found matching the requested type.
     */
    MobileResource getResourceByType(Resource resource, String type, boolean deep);

    /**
     * Traverse through parent resources to locate a type matching the type parameter.  If found, its resource
     * will be returned, otherwise <code>null</code> will be returned.  Processing will also stop if any types specified
     * in the <code>stopTypes</code> array are found first.
     * @param resource The resource to start searching from.
     * @param type The mobile resource type id. Use {@link com.adobe.cq.mobile.platform.MobileResourceType} for common app types.
     * @param stopTypes An array of types that will stop processing.
     * @return A {@link MobileResource} matching the found type.
     */
    MobileResource findAncestorResourceByType(Resource resource, String type, String... stopTypes);

    /**
     * Locate the closest mobile resource matching to specified type. This mobile resource will be compared
     * first before its ancestors are.  A value of <code>null</code> will be returned
     * if no match is found.
     * @param resource The resource to start searching from.
     * @param type The mobile resource type id. Use {@link com.adobe.cq.mobile.platform.MobileResourceType} for common app types.
     * @param stopTypes An array of types that will stop processing.
     * @return A {@link MobileResource} of the closest matching type, if any exist.
     */
    MobileResource findClosestResourceByType(Resource resource, String type, String... stopTypes);
}
