/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.scheduled.exporter.process;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Service;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.day.cq.commons.jcr.JcrUtil;
import com.day.cq.workflow.WorkflowException;
import com.day.cq.workflow.WorkflowSession;
import com.day.cq.workflow.exec.WorkItem;
import com.day.cq.workflow.exec.WorkflowProcess;
import com.day.cq.workflow.metadata.MetaDataMap;

/**
 * <p>
 * Extracts the payload path and saves it as a sling:Folder property
 * {@link ExtractExportDataProcess#PN_PAYLOAD} below a a predefined
 * <code>destPath</code> provided as process argument. The sling:Folder node
 * name is a SHA-1 hexadecimal representation of the payload path. The node will
 * be created only if it does not exist already.
 * </p>
 * <p>
 * Process is only executed if the <code>destPath</code> argument is provided
 * and the payload type is JCR_PATH.
 * </p>
 * <b>Arguments:</b>
 * <table>
 * <tr>
 * <td><b>Prefix</b></td>
 * <td><b>Example</b></td>
 * </tr>
 * <tbody>
 * <tr>
 * <td>destPath</td>
 * <td>destPath:/var/dataexport/example</td>
 * </tr>
 * </tbody>
 * </table>
 */
@Component
@Service
@Property(name = "process.label", value = "Extract Export Data")
public class ExtractExportDataProcess implements WorkflowProcess {

    private final Logger log = LoggerFactory.getLogger(this.getClass());

    public static final String PN_PAYLOAD = "payload";

    private static final String NT_SLINGFOLDER = "sling:Folder";

    private static final String TYPE_JCR_PATH = "JCR_PATH";

    /**
     * The available arguments to this process implementation.
     */
    public enum Arguments {
        PROCESS_ARGS("PROCESS_ARGS"), DEST_PATH("destPath");

        private String argumentName;

        Arguments(String argumentName) {
            this.argumentName = argumentName;
        }

        public String getArgumentName() {
            return this.argumentName;
        }

        public String getArgumentPrefix() {
            return this.argumentName + ":";
        }

    }

    /*
     * (non-Javadoc)
     * @see
     * com.day.cq.workflow.exec.WorkflowProcess#execute(com.day.cq.workflow.
     * exec.WorkItem, com.day.cq.workflow.WorkflowSession,
     * com.day.cq.workflow.metadata.MetaDataMap)
     */
    public void execute(WorkItem item, WorkflowSession workflowSession, MetaDataMap metaData) throws WorkflowException {
        String[] args = buildArguments(metaData);
        try {
            if (TYPE_JCR_PATH.equals(item.getWorkflowData().getPayloadType())) {
                final Session session = workflowSession.getSession();
                String destPath = null;
                if (getValuesFromArgs(Arguments.DEST_PATH.getArgumentName(), args).size() > 0) {
                    destPath = getValuesFromArgs(Arguments.DEST_PATH.getArgumentName(), args).get(0);
                }
                if (destPath == null) {
                    log.error("Argument destPath missing.");
                    return;
                }
                final String itemPath = item.getWorkflowData().getPayload().toString();
                final String nodeName = DigestUtils.shaHex(itemPath.getBytes());

                if (!session.nodeExists(destPath + "/" + nodeName)) {
                    Node pathNode = JcrUtil.createPath(destPath + "/" + nodeName, NT_SLINGFOLDER, session);
                    pathNode.setProperty(PN_PAYLOAD, itemPath);
                }

                if (session.hasPendingChanges()) {
                    session.save();
                }
            }
        } catch (RepositoryException e) {
            log.error("execute: error while extracting data; work item [{}]: ", item.getId(), e);
        }
    }

    /**
     * Builds an array of arguments out of the provided <code>MetaDataMap</code>
     * 
     * @param metaData
     * @return
     */
    String[] buildArguments(MetaDataMap metaData) {
        String processArgs = metaData.get(Arguments.PROCESS_ARGS.name(), String.class);
        if (processArgs != null && !processArgs.equals("")) {
            return processArgs.split(",");
        } else {
            List<String> arguments = new ArrayList<String>();
            String targetpath = metaData.get(Arguments.DEST_PATH.name(), String.class);
            if (StringUtils.isNotBlank(targetpath)) {
                arguments.add(Arguments.DEST_PATH.getArgumentPrefix() + targetpath);
            }

            return arguments.toArray(new String[arguments.size()]);
        }
    }

    /**
     * Returns the values for a specific key from the provided arguments.
     * 
     * @param key Argument key
     * @param arguments Arguments
     * @return List of values
     */
    private List<String> getValuesFromArgs(String key, String arguments[]) {
        final List<String> values = new LinkedList<String>();
        for (String str : arguments) {
            if (str.startsWith(key + ":")) {
                final String mt = str.substring((key + ":").length()).trim();
                values.add(mt);
            }
        }
        return values;
    }

}
