/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 ************************************************************************/
package com.adobe.cq.screens.device;

import java.util.Iterator;

import aQute.bnd.annotation.ProviderType;
import org.apache.commons.collections.Predicate;
import org.apache.sling.commons.json.JSONException;
import org.apache.sling.commons.json.JSONObject;

import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.jcr.RepositoryException;

/**
 * Manages the devices in the repository.
 */
@ProviderType
public interface DeviceManager {

    /**
     * Returns the device with the given id or {@code null}
     * @param deviceId the device id
     * @return the device or {@code null}
     */
    @CheckForNull
    Device getDevice(@Nonnull String deviceId);

    /**
     * Returns the status of the device with the given id.
     * @param deviceId the device id
     * @return status or {@code null} if the device does not exist.
     */
    @CheckForNull
    DeviceStatus getDeviceStatus(@Nonnull String deviceId);

    /**
     * Returns an iterator over all devices.
     * @return an iterator.
     */
    @Nonnull
    Iterator<Device> getDevices();

    /**
     * Returns an iterator over all devices that fulfill the given predicate.
     * @param predicate the filter predicate or {@code null}
     * @return the iterator.
     */
    @Nonnull
    Iterator<Device> getDevices(@Nullable Predicate predicate);

    /**
     * Creates a new device for the given tenant.
     * @param tenant the name of the tenant. eg "geometrixx".
     * @return the newly created device
     * @throws RepositoryException if an error occurrs.
     */
    @Nonnull
    Device createDevice(@Nonnull String tenant) throws RepositoryException;

    /**
     * Deletes an existing device, and removes any existing assignment to a display.
     * @param deviceId the device id
     * @throws RepositoryException if an error occurrs.
     */
    void deleteDevice(@Nonnull String deviceId) throws RepositoryException;

    /**
     * Generates and sets a new password for the device user.
     * @param device the device
     * @return the new password
     * @throws RepositoryException if an error occurrs.
     */
    @Nonnull
    String updatePassword(@Nonnull Device device) throws RepositoryException;

    /**
     * Updates the device metadata.
     * @param device the device
     * @param metadata the metadata.
     * @return the device for chaining.
     * @throws JSONException if a JSON error occurrs
     * @throws RepositoryException if an error occurrs
     */
    @Nonnull
    Device setMetadata(@Nonnull Device device, @Nonnull JSONObject metadata) throws JSONException, RepositoryException;

    /**
     * Assigns a display to a device and creates a new DeviceConfig.
     *
     * @param device the device
     * @param displayPath the display path
     * @return the device for chaining.
     * @throws RepositoryException if an error occurrs.
     */
    @Nonnull
    Device assignDisplay(@Nonnull Device device, @Nonnull String displayPath) throws RepositoryException;

    /**
     * Assigns a device to an existing device config.
     *
     * @param device the device
     * @param config the device config
     * @return the device for chaining.
     * @throws RepositoryException if an error occurrs.
     */
    @Nonnull
    Device assignDeviceConfig(@Nonnull Device device, @Nonnull DeviceConfig config) throws RepositoryException;

    /**
     * Removes the display assignment from the device.
     * @param device the device
     * @param removeConfig {@code true} to also remove the device config from the display.4
     * @return the device for chaining.
     * @throws RepositoryException if an error occurrs.
     */
    @Nonnull
    Device removeDisplay(@Nonnull Device device, boolean removeConfig) throws RepositoryException;

}
