/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.activitystreams.listener.api;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Reference;
import org.apache.sling.api.resource.LoginException;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceResolverFactory;
import org.apache.sling.jcr.api.SlingRepository;
import org.apache.sling.jcr.resource.JcrResourceConstants;
import org.osgi.service.component.ComponentContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.cq.social.activitystreams.api.SocialActivityManager;
import com.adobe.cq.social.activitystreams.impl.Activator;
import com.adobe.granite.activitystreams.Activity;
import com.adobe.granite.activitystreams.ActivityException;
import com.adobe.granite.activitystreams.ActivityStream;

/**
 * Base class of the ActivityStreamProvider.
 */
@Component(componentAbstract = true, label = "%abstractactivitystreamprovider.name")
public abstract class AbstractActivityStreamProvider implements ActivityStreamProvider {
    protected static final String UGC_WRITER = "ugc-writer";

    /**
     * Eventually remove SLING_SERVICE_SUBSERVICE once we can update the dependencies to get this from
     * ResourceResolverFactory and not make the unit tests fail.
     */
    protected static final String SLING_SERVICE_SUBSERVICE = "sling.service.subservice";

    /**
     * The logger.
     */
    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractActivityStreamProvider.class);

    /**
     * Reference to ResourceResolverFactory. Can't remove this it is API, but internally it is not used.
     */
    @Reference
    protected ResourceResolverFactory resourceResolverFactory;
    @Reference
    protected SocialActivityManager activityManager;

    @Reference
    protected final SlingRepository repository = null;

    /**
     * Because this is an abstract component we can't just add @Reference without bumping the package API major
     * version. So we avoid using Declarative Services and just use the next layer down directly.
     */
    private com.adobe.cq.social.serviceusers.internal.ServiceUserWrapper serviceUserWrapper;
    /**
     * Using this one lets us configure the service user with the symbolic name of *this* bundle, not the symbolic
     * name of each bundle implementing a concrete version of this component.
     */
    private ResourceResolverFactory privateResourceResolverFactory;

    public com.adobe.cq.social.serviceusers.internal.ServiceUserWrapper getServiceUserWrapper() {
        if (serviceUserWrapper == null) {
            serviceUserWrapper =
                Activator.getService(com.adobe.cq.social.serviceusers.internal.ServiceUserWrapper.class);
        }
        return serviceUserWrapper;
    }

    public ResourceResolverFactory getPrivateResourceResolverFactory() {
        if (privateResourceResolverFactory == null) {
            privateResourceResolverFactory = Activator.getService(ResourceResolverFactory.class);
        }
        return privateResourceResolverFactory;
    }

    @Override
    public void append(final Activity activity) throws ActivityException {

        if (activityManager == null) {
            return;
        }
        ResourceResolver ugcResolver = null;
        try {
            ugcResolver =
                getServiceUserWrapper().getServiceResourceResolver(getPrivateResourceResolverFactory(),
                    Collections.singletonMap(SLING_SERVICE_SUBSERVICE, (Object) UGC_WRITER));
            final ActivityStream stream = getStream(activity, ugcResolver);
            if (stream != null) {
                stream.append(activity);
            }
        } catch (final LoginException le) {
            LOGGER.error("Can't obtain resolver for service user", le);
            return;
        } finally {
            if (ugcResolver != null && ugcResolver.isLive()) {
                ugcResolver.close();
            }
        }
    }

    protected abstract ActivityStream getStream(Activity activity, ResourceResolver resolver)
        throws ActivityException;

    /**
     * Get the session to be used by the stream
     * @return the admin session or null if there is an error.
     * @throws RepositoryException
     */
    protected abstract Session getAdminSession() throws RepositoryException;

    /**
     * Activate this service. Subclass component needs to call this method in its activate method.
     * @param ctx
     */
    protected void activate(final ComponentContext ctx) throws LoginException {
    }

    /**
     * Deactivate this service. Subclass component needs to call this method in its deactivate method.
     * @param context OSGi component context
     */
    protected void deactivate(final ComponentContext context) {
        activityManager = null;
    }

    protected ResourceResolver getServiceResourceResolver(final Session session) throws LoginException {
        final Map<String, Object> authInfo = new HashMap<String, Object>();
        authInfo.put(JcrResourceConstants.AUTHENTICATION_INFO_SESSION, session);
        return resourceResolverFactory.getResourceResolver(authInfo);
    }
}
