/**************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2015 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.badging.api;

import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.jcr.RepositoryException;

import org.apache.sling.api.resource.PersistenceException;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;

import com.adobe.cq.social.community.api.CommunityContext;

/**
 * Badging service exposes operations like getBadges needed by components.
 */
public interface BadgingService {
    /**
     * badging engine type.
     */
    String BADGING_TYPE_PROP = "badgingType";
    /**
     * badging rule property.
     */
    String BADGING_RULES_PROP = "badgingRules";
    /**
     * badging folder name located in SRP after the home users profile dir.
     */
    String BADGING_FOLDER_NAME = "badges";
    /**
     * The containing page path of the rule's location this badge belongs to.
     */
    String RULE_PAGE_PATH_PROP = "rulePage_s";
    /**
     * Path of the badge's resource/image.
     */
    String BADGE_PATH_PROP = "badgeContentPath";
    /**
     * The site the badge belongs to.
     */
    String SITE_ID_PROP = "siteId_s";
    /**
     * The rule path.
     */
    String RULE_PROP = "badgingRule";
    /**
     * True if this is an assigned badge.
     */
    String ASSIGNED_PROP = "assigned_b";
    /**
     * The score when badge was awarded.
     */
    String BADGE_EARNED_SCORE_PROP = "earnedScore";
    /**
     * The date when badge was awarded.
     */
    String BADGE_EARNED_DATE_PROP = "earnedDate_dt";

    /**
     * Badging types.
     */
    String ALL_BADGES = "ALL";
    String ASSIGNED_BADGES = "ASSIGNED";
    String EARNED_BADGES = "EARNED";

    /**
     * basic badging engine specific properties.
     */
    String THRESHOLDS_PROP = "thresholds";
    String THRESHOLDS_SEPARATOR = "|";

    /**
     * advanced badging engine specific properties
     */
    String BADGE_PATHS_PROP = "badgePath";
    String BADGING_LEVELS_PROP = "badgingLevels";

    /**
     * For components supporting badge display, use this Boolean property to toggle badge display on/off.
     */
    String ALLOW_BADGES_PROP = "allowBadges";

    /**
     * When badge display is turned on in a component, set this boolean property in the
     * component resource to display all the user's badges or just badges earned/assigned for this component. The
     * default is set to false so set this to true to display badges earned at this component or any components above
     * this component.
     */
    public static final String FILTER_BY_COMP_PROP = "filterBadgesByComponent";

    /**
     * Retrieves badges for a user.
     * @param resourceResolver resolver used for reading badge resources
     * @param userId the user's userid
     * @param pageResource restrict badges that are in this resource (and above) only. null for no path restrictions.
     *            Note that a resource in a community restricts badges to that community so no need to set the context
     *            parameter.
     * @param context restrict badges to this community. null for no community restrictions. Useful for retrieving all
     *            badges for a specific community by setting pageResource to null.
     * @param badgeType the badge type to retrieve: ALL, ASSIGNED, EARNED
     * @return list of {@link UserBadge} objects for the user.
     * @throws RepositoryException thrown if unable to retrieve the score
     */
    List<UserBadge> getBadges(@Nonnull ResourceResolver resourceResolver, @Nonnull String userId,
        Resource pageResource, CommunityContext context, String badgeType) throws RepositoryException;

    /**
     * Save or update a user's badge.
     * @param resolver resolver used for writing the badge
     * @param userId the user's userid
     * @param ruleResource the badging rule the badge was earned from. Can't be null for earned badges.
     * @param componentResource the resource where the rule was located (Not necessarily at a component. Could be at
     *            jcr:content, etc.). Can't be null for earned badges.
     * @param badgeContentPath the path for the badge content resource containing the image
     * @param assigned true if this badge is an assigned badge, false for an earned badge
     * @param props any addition properties to store with the badge
     * @throws RepositoryException error saving badge
     * @throws PersistenceException error saving badge
     */
    void saveBadge(@Nonnull ResourceResolver resolver, @Nonnull String userId, Resource ruleResource,
        Resource componentResource, @Nonnull String badgeContentPath, @Nonnull boolean assigned,
        Map<String, Object> props) throws RepositoryException, PersistenceException;

    /**
     * Delete a user's badges where the badgeContentPath matches the input parameter.
     * @param resolver resolver used for writing the badge
     * @param userId the user's userid
     * @param ruleResource the badging rule the badge was earned from. Can't be null for earned badges.
     * @param componentResource the resource where the rule was located (Not necessarily at a component. Could be at
     *            jcr:content, etc.). Can't be null for earned badges.
     * @param badgeContentPath the path for the badge content resource containing the image
     * @param assigned true for only assigned badges
     * @return number of badges deleted
     * @throws RepositoryException error saving badge
     * @throws PersistenceException error saving badge
     */
    int deleteBadge(@Nonnull ResourceResolver resolver, @Nonnull String userId, Resource ruleResource,
        Resource componentResource, @Nonnull String badgeContentPath, boolean assigned) throws RepositoryException,
        PersistenceException;
}
