/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 *
 *************************************************************************/
package com.adobe.cq.social.calendar.client.api;

import java.io.Serializable;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.jcr.InvalidItemStateException;

import com.adobe.cq.social.scf.SocialComponent;
import com.adobe.cq.social.scf.core.SocialEvent;
import com.adobe.granite.activitystreams.ObjectTypes;
import com.adobe.granite.activitystreams.Verbs;

/**
 * Event fired for changes occurring within a calendar.
 */
public class CalendarActionsEvent extends SocialEvent<CalendarActionsEvent.CalendarActions> implements Serializable {
    /**
     * The event topic suffix for Calendar events.
     */
    public static final String CALENDAR_TOPIC = "calendar";

    /** A serialVersionUID for this class. */
    private static final long serialVersionUID = 2L;

    /**
     * @param path - the jcr path to either the question node or the answer node on which the event was triggered.
     * @param userId - the user id of the user who triggered the event.
     * @param action - the action that was taken to trigger the event.
     * @param additionalData - any additional data that is part of the event payload.
     */
    public CalendarActionsEvent(final String path, final String userId, final CalendarActions action,
        final Map<String, Object> additionalData) {
        super(CALENDAR_TOPIC, path, userId, action, additionalData);
    }

    /**
     * The event types supported for the forum and its topics/posts.
     */
    public enum CalendarActions implements com.adobe.cq.social.scf.core.SocialEvent.SocialActions {
        /**
         * The event fired upon a calendar event (topic) being added.
         */
        CalendarEventAdded,

        /**
         * The event fired upon a calendar event reply (post) being added.
         */
        CalendarEventReplyAdded,

        /**
         * The event fired upon a calendar event (topic) being edited.
         */
        CalendarEventEdited,

        /**
         * The event fired upon a calendar event reply (post) being edited.
         */
        CalendarEventReplyEdited,

        /**
         * The event fired upon a calendar event (topic) being deleted.
         */
        CalendarEventDeleted,

        /**
         * The event fired upon a calendar event reply (post) being deleted.
         */
        CalendarEventReplyDeleted;

        @Override
        public String getVerb() {
            switch (this) {
                case CalendarEventAdded:
                    return Verbs.POST;
                case CalendarEventReplyAdded:
                    return Verbs.ADD;
                case CalendarEventEdited:
                    return Verbs.UPDATE;
                case CalendarEventReplyEdited:
                    return Verbs.UPDATE;
                case CalendarEventDeleted:
                    return Verbs.DELETE;
                case CalendarEventReplyDeleted:
                    return Verbs.DELETE;
                default:
                    throw new IllegalArgumentException();
            }
        }
    }

    /**
     * Creates a CalendarActionsEvent that represents an action of {@link CalendarActions#CalendarEventAdded}.
     * @param calendarEvent - the {@link CalendarEvent} that was created
     * @param userId - the user id of the user who created the {@link CalendarEvent}.
     * @return the event
     */
    public static CalendarActionsEvent calendarEventCreated(final CalendarEvent calendarEvent, final String userId) {
        return buildEventForCalendarEvent(calendarEvent, userId, CalendarActions.CalendarEventAdded);
    }

    /**
     * Creates a CalendarActionsEvent that represents an action of {@link CalendarActions#CalendarEventEdited}.
     * @param calendarEvent - the {@link CalendarEvent} that was edited
     * @param userId - the user id of the user who edited the {@link CalendarEvent}.
     * @return the event
     */
    public static CalendarActionsEvent calendarEventUpdated(final CalendarEvent calendarEvent, final String userId) {
        return buildEventForCalendarEvent(calendarEvent, userId, CalendarActions.CalendarEventEdited);
    }

    /**
     * Creates a CalendarActionsEvent that represents an action of {@link CalendarActions#CalendarEventDeleted}.
     * @param calendarEvent - the {@link CalendarEvent} that was deleted
     * @param userId - the user id of the user who deleted the {@link CalendarEvent}.
     * @return the event
     */
    public static CalendarActionsEvent calendarEventDeleted(final CalendarEvent calendarEvent, final String userId) {
        return buildEventForCalendarEvent(calendarEvent, userId, CalendarActions.CalendarEventDeleted);
    }

    /**
     * Creates a CalendarActionsEvent that represents an action of {@link CalendarActions#CalendarEventReplyAdded}.
     * @param calendarEvent - the {@link CalendarEvent} (reply on event) that was created
     * @param userId - the user id of the user who created the {@link CalendarEvent} (reply on event).
     * @return the event
     */
    public static CalendarActionsEvent calendarEventReplyCreated(final CalendarEvent calendarEvent,
        final String userId) {
        return buildEventForCalendarEventReply(calendarEvent, userId, CalendarActions.CalendarEventReplyAdded);
    }

    /**
     * Creates a CalendarActionsEvent that represents an action of {@link CalendarActions#CalendarEventReplyEdited}.
     * @param calendarEvent - the {@link CalendarEvent} (reply on event) that was edited
     * @param userId - the user id of the user who edited the {@link CalendarEvent} (reply on event).
     * @return the event
     */
    public static CalendarActionsEvent calendarEventReplyUpdated(final CalendarEvent calendarEvent,
        final String userId) {
        return buildEventForCalendarEventReply(calendarEvent, userId, CalendarActions.CalendarEventReplyEdited);
    }

    /**
     * Creates a CalendarActionsEvent that represents an action of {@link CalendarActions#CalendarEventReplyDeleted}.
     * @param calendarEvent - the {@link CalendarEvent} (reply on event) that was deleted
     * @param userId - the user id of the user who deleted the {@link CalendarEvent} (reply on event).
     * @return the event
     */
    public static CalendarActionsEvent calendarEventReplyDeleted(final CalendarEvent calendarEvent,
        final String userId) {
        return buildEventForCalendarEventReply(calendarEvent, userId, CalendarActions.CalendarEventReplyDeleted);
    }

    private static CalendarActionsEvent buildEventForCalendarEvent(final CalendarEvent calendarEvent,
        final String userId, final CalendarActions action) {
        final Map<String, Object> addData = new HashMap<String, Object>(2);
        addData.put(OBJECT, new BaseEventObject(getDisplayName(calendarEvent), calendarEvent.getResource().getPath(),
            ObjectTypes.ARTICLE));
        addData.put(
            TARGET,
            new BaseEventObject(getDisplayName(calendarEvent.getParentComponent()), calendarEvent
                .getSourceComponentId(), ObjectTypes.COLLECTION));
        return new CalendarActionsEvent(calendarEvent.getResource().getPath(), userId, action, addData);
    }

    private static CalendarActionsEvent buildEventForCalendarEventReply(final CalendarEvent calendarEvent,
        final String userId, final CalendarActions action) {
        final Map<String, Object> addData = new HashMap<String, Object>(2);
        if (calendarEvent.isTopic()) {
            // Event
            addData.put(OBJECT, new BaseEventObject(getDisplayName(calendarEvent), calendarEvent.getResource()
                .getPath(), ObjectTypes.ARTICLE));
            addData.put(
                TARGET,
                new BaseEventObject(getDisplayName(calendarEvent.getSourceComponent()), calendarEvent
                    .getSourceComponentId(), ObjectTypes.COLLECTION));
        } else {
            // EventReply
            addData.put(TARGET,
                new BaseEventObject(getDisplayName(calendarEvent.getParentComponent()), calendarEvent.getParentId(),
                    ObjectTypes.ARTICLE));
            addData.put(OBJECT, new BaseEventObject(getDisplayName(calendarEvent), calendarEvent.getResource()
                .getPath(), ObjectTypes.COMMENT));
        }

        return new CalendarActionsEvent(calendarEvent.getResource().getPath(), userId, action, addData);
    }

    private static String getDisplayName(final SocialComponent socialComponent) {
        if (socialComponent instanceof CalendarEvent) {
            String defaultName =
                ((CalendarEvent) socialComponent).isTopic() ? "a calendar event" : "a calendar event reply";
            try {
                final String name = ((CalendarEvent) socialComponent).getSubject();
                if (name != null) {
                    return name;
                } else {
                    return defaultName;
                }
            } catch (final IllegalArgumentException e) {
                if (e.getCause() instanceof InvalidItemStateException) {
                    return defaultName;
                } else {
                    throw e;
                }
            }
        } else {
            if (socialComponent instanceof Calendar) {
                return ((Calendar) socialComponent).getName();
            } else {
                return null;
            }
        }
    }
}
