/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 *
 *************************************************************************/
package com.adobe.cq.social.calendar.client.api;

import com.day.cq.commons.date.DateUtil;
import com.day.cq.commons.date.InvalidDateException;

/**
 * Calendar DateRangeFilter which maintains the upper and lower bounded date range of a Calendar's event listing. A
 * date range can be specified as a selector in the format of: <li>lower-bounded_upper-bounded where 'lower-bounded'
 * and 'upper-bounded' are the starting date and ending date.</li> <li>starting-date_ND, where 'N' is an integer
 * specified the number of day after the starting date</li> <li>starting-date_NM, where 'N' is an integer specified
 * the number of month after the starting date.</li> <li>starting-date_NY, where 'Y' is an integer specified the
 * number of year after the starting date.</li>
 */
public interface DateRangeFilter {
    enum DateRangeType {
        DAYOFFSET("D", java.util.Calendar.DAY_OF_MONTH), WEEKOFFSET("W", java.util.Calendar.WEEK_OF_MONTH), MONTHOFFSET(
            "M", java.util.Calendar.MONTH), YEAROFFSET("Y", java.util.Calendar.YEAR), RANGE("",
            java.util.Calendar.DAY_OF_MONTH);

        final String suffix;
        final int calendarFieldNumber;

        /**
         * Constructor
         * @param suffix the request suffix for the date range type
         * @param calendarFieldNumber the equivalent of Java.util.Calendar constant
         */
        DateRangeType(final String suffix, final int calendarFieldNumber) {
            this.suffix = suffix;
            this.calendarFieldNumber = calendarFieldNumber;
        }

        public String getSuffix() {
            return suffix;
        }

        public int getCalendarFieldNumber() {
            return calendarFieldNumber;
        }
    }

    /**
     * Represent the upper bound specification
     */
    class UpperBoundedValue {
        public static UpperBoundedValue ONE_MONTH_OFFSET;
        static {
            try {
                ONE_MONTH_OFFSET = new UpperBoundedValue("1M");
            } catch (final InvalidDateException ignored) {
            }
        }

        final DateRangeType type;
        final int offset; // the specified offset for DAY, WEEK, MONTH, and YEAR type
        java.util.Calendar cal;  // the end date for the RANGE type
        final String value;

        public UpperBoundedValue(final String value) throws InvalidDateException {
            this.value = value;
            type = getDateRangeFilter(value);
            if (type != DateRangeType.RANGE) {
                final String s = value.substring(0, value.length() - 1);
                try {
                    offset = Integer.parseInt(s);
                    cal = null;
                } catch (final NumberFormatException e) {
                    throw new InvalidDateException("Invalid date range: " + s);
                }
            } else {
                offset = -1;
            }
        }

        /**
         * Get the type of this instance
         * @return the DateRangeType
         */
        public DateRangeType getType() {
            return type;
        }

        /**
         * Get the value of this instance.
         * @return this instance value
         */
        public String value() {
            return value;
        }

        /**
         * Get the original offset if the DateRangeFilter is not of type <code>DateRangeType.RANGE</code>
         */
        public int getOffset() {
            if (type != DateRangeType.RANGE) {
                return offset;
            } else {
                throw new UnsupportedOperationException("getOffset is not supported for DateRangeType.RANGE");
            }
        }

        /**
         * Get the value of this instance. If this is a <code>RANGE</code> type, the method returns the client
         * specification value, otherwise, it computes the value base on the client offset specification.
         * @param start the start date
         * @return
         */
        public java.util.Calendar getCalendar(final java.util.Calendar start) {
            if (type != DateRangeType.RANGE) {
                final java.util.Calendar retVal = (java.util.Calendar) start.clone();
                retVal.add(type.getCalendarFieldNumber(), offset);
                return retVal;
            } else {
                return cal;
            }
        }

        /**
         * Get the offset <code>Calendar</code> from the specified start date and offset.
         * @param start the calendar which we apply the offset to
         * @param offset the offset, which can be positive or negative integer.
         * @return the offset Calendar.
         */
        public java.util.Calendar getOffset(final java.util.Calendar start, final int offset) {
            final java.util.Calendar retVal = (java.util.Calendar) start.clone();
            retVal.add(type.getCalendarFieldNumber(), offset);
            return retVal;
        }

        // TODO: Make sure the 2 calendars are in the same time zone
        public int daysBetween(final java.util.Calendar c1, final java.util.Calendar c2) {
            return (int) ((c2.getTimeInMillis() - c1.getTimeInMillis()) / (1000 * 60 * 60 * 24));
        }

        private DateRangeType getDateRangeFilter(final String date) throws InvalidDateException {
            final String suffix = "" + date.charAt(date.length() - 1);
            if (suffix.equals(DateRangeType.DAYOFFSET.getSuffix())) {
                return DateRangeType.DAYOFFSET;
            }
            if (suffix.equals(DateRangeType.WEEKOFFSET.getSuffix())) {
                return DateRangeType.WEEKOFFSET;
            }
            if (suffix.equals(DateRangeType.MONTHOFFSET.getSuffix())) {
                return DateRangeType.MONTHOFFSET;
            }
            if (suffix.equals(DateRangeType.YEAROFFSET.getSuffix())) {
                return DateRangeType.YEAROFFSET;
            }
            // parse the input to make sure that it is a date
            cal = DateUtil.parseISO8601(date);
            return DateRangeType.RANGE;
        }
    }

    /**
     * Get the start date and time. This uses the ISO8601 format for dates (<code>YYYY-MM-DDTHH:mm:ss.SSSZ</code>).
     * @return a calendar specifies the start date and time.
     */
    String getLowerBound();

    /**
     * Get the lower bound calendar
     */
    java.util.Calendar getLowerBoundCalendar();

    /**
     * Get the upperBounded value.
     */
    UpperBoundedValue getUpperBoundedValue();

    /**
     * Get the upper bound calendar
     */
    java.util.Calendar getUpperBoundCalendar();

    /**
     * Is the start date include in the filter.
     * @return <code>true</code> if the start date and time is included in the filter
     */
    boolean includeLowerBoundValue();

    /**
     * Is the end date include in the filter.
     * @return <code>true</code> if the end date and time is included in the filter.
     */
    boolean includeUpperBoundValue();
}
