/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.commons.comments.api;

import javax.annotation.Nonnull;

import org.apache.commons.lang3.StringUtils;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceUtil;
import org.apache.sling.api.resource.ValueMap;

import com.adobe.cq.social.scf.CollectionPagination;
import com.adobe.cq.social.scf.core.CollectionSortedOrder;
import com.fasterxml.jackson.annotation.JsonProperty;

/**
 * Base implementation of the {@link CommentCollectionConfiguration}.
 */
public class AbstractCommentCollectionConfiguration implements CommentCollectionConfiguration {

    private boolean allowsVoting;
    private boolean isRTEEnabled;
    private boolean allowsAttachment;
    private final boolean allowsTranslateAllButton;
    private final boolean allowsFlagging;
    private final boolean allowsClose;
    private final boolean allowsDeny;
    private final boolean allowsEdit;
    private final boolean allowsReply;
    private final boolean allowsFollowing;
    private boolean allowsDelete;
    private boolean allowsMove;
    private boolean allowsCustomFlagReason = true;
    private int pageSize = 10;
    private final ValueMap configValueMap;
    private final CollectionSortedOrder sortOrder;

    /**
     * Constructor.
     * @param resource the target resource of the collection or comment
     * @param cs the {@link CommentSystem} instance of the CommentCollection
     */
    public AbstractCommentCollectionConfiguration(final Resource commentSystem) {
        this(ResourceUtil.getValueMap(commentSystem));
    }

    /**
     * Constructor.
     * @param valuemap properties valuemap to use for the configuration
     */
    public AbstractCommentCollectionConfiguration(@Nonnull final ValueMap vm) {
        allowsVoting =
            vm.get(CommentCollectionConfiguration.PN_ALLOW_VOTING,
                CommentCollectionConfiguration.PV_DEFAULT_ALLOW_VOTING);
        allowsEdit = true;// cs.allowsEdit();

        pageSize = vm.get(PN_MAX_ITEMS_PER_PAGE, CollectionPagination.DEFAULT_PAGE_SIZE);
        allowsTranslateAllButton = vm.get(PN_ALLOW_TRANSLATE_ALL, PV_DEFAULT_ALLOW_TRANSLATE_ALL);
        allowsCustomFlagReason = vm.get(PN_ALLOW_CUSTOM_FLAGREASON, PV_DEFAULT_ALLOW_CUSTOM_FLAGREASON);
        allowsAttachment = vm.get(PN_ALLOW_ATTACHMENT, PV_DEFAULT_ALLOW_ATTACHMENT);
        isRTEEnabled = vm.get(PN_RTE_ENABLED, PV_DEFAULT_RTE_ENABLED);
        allowsDeny = vm.get(PN_ALLOW_DENY, PV_DEFAULT_ALLOW_DENY);
        allowsFlagging = vm.get(PN_ALLOW_FLAGGING, PV_DEFAULT_ALLOW_FLAGGING);
        allowsFollowing = vm.get(PN_ALLOW_FOLLOWING, PV_DEFAULT_ALLOW_FOLLOWING);
        allowsReply = vm.get(PN_ALLOW_REPLIES, PV_DEFAULT_ALLOW_REPLIES);
        allowsMove = vm.get(PN_ALLOW_MOVE, PV_DEFAULT_ALLOW_MOVE);
        allowsClose = vm.get(PN_ALLOW_CLOSE, PV_DEFAULT_ALLOW_CLOSE);
        allowsDelete = vm.get(PN_ALLOW_DELETE_COMMENTS, PV_DEFAULT_ALLOW_DELETE_COMMENTS);
        final String sortOrderValue = vm.get(PN_SORT_ORDER, "DESC");
        this.sortOrder =
            StringUtils.equalsIgnoreCase("asc", sortOrderValue) ? CollectionSortedOrder.DEFAULT_ORDER
                : CollectionSortedOrder.REVERSED_ORDER;
        this.configValueMap = vm;
    }

    @Override
    @JsonProperty("isVotingAllowed")
    public boolean isVotingAllowed() {
        return allowsVoting;
    }

    @Override
    @JsonProperty("isAttachmentAllowed")
    public boolean isAttachmentAllowed() {
        return allowsAttachment;
    }

    @Override
    @JsonProperty("isRTEEnabled")
    public boolean isRTEEnabled() {
        return isRTEEnabled;
    }

    @Override
    @JsonProperty("isCustomFlagReasonAllowed")
    public boolean isCustomFlagReasonAllowed() {
        return allowsCustomFlagReason;
    }

    @Override
    @JsonProperty("isTranslateAllButtonAllowed")
    public boolean isTranslateAllButtonAllowed() {
        return allowsTranslateAllButton;
    }

    @Override
    public int getPageSize() {
        return this.pageSize;
    }

    @Override
    @JsonProperty("isFlaggingAllowed")
    public boolean isFlaggingAllowed() {
        return allowsFlagging;
    }

    @Override
    @JsonProperty("isCloseAllowed")
    public boolean isCloseAllowed() {
        return allowsClose;
    }

    @Override
    @JsonProperty("isDenyAllowed")
    public boolean isDenyAllowed() {
        return allowsDeny;
    }

    @Override
    @JsonProperty("isEditAllowed")
    public boolean isEditAllowed() {
        return allowsEdit;
    }

    @Override
    @JsonProperty("isReplyAllowed")
    public boolean isReplyAllowed() {
        return allowsReply;
    }

    @Override
    @JsonProperty("isDeleteAllowed")
    public boolean isDeleteAllowed() {
        return allowsDelete;
    }

    protected void setAllowsMove(final boolean allowsMove) {
        this.allowsMove = allowsMove;
    }

    protected void setAllowsDelete(final boolean flag) {
        allowsDelete = flag;
    }

    protected void setAllowsAttachment(final boolean flag) {
        allowsAttachment = flag;
    }

    protected void setAllowsVoting(final boolean flag) {
        allowsVoting = flag;
    }

    protected void setRTEEnabled(final boolean flag) {
        isRTEEnabled = flag;
    }

    /**
     * Gets the configuration value map that this configuration object is based off of.
     * @return a value map that has configuration details in it.
     */
    @Nonnull
    protected ValueMap getConfigValueMap() {
        return this.configValueMap;
    }

    @Override
    public boolean isFollowingAllowed() {
        return allowsFollowing;
    }

    @Override
    public CollectionSortedOrder getSortOrder() {
        return this.sortOrder;
    }

    @Override
    public boolean isMoveAllowed() {
        return allowsMove;
    }

}
