/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.commons.comments.api;

import java.util.Calendar;
import java.util.List;
import java.util.Map;

import com.adobe.cq.social.commons.Attachment;
import com.adobe.cq.social.commons.moderation.api.FlagReason;
import com.adobe.cq.social.scf.SocialCollectionComponent;
import com.adobe.cq.social.scf.SocialComponent;
import com.adobe.cq.social.scf.User;
import com.adobe.cq.social.scf.core.ResourceID;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.annotation.JsonProperty;

/**
 * <p>
 * A <code>Comment</code> represents a comment created by a user (author) on a specific target (e.g. a page, asset or
 * any node for that matter). The comments are contained within a {@link CommentCollection} (see {@link #getParent()}.
 * A comment simply provides a comment text or "message" (see {@link #getMessage()} and access to various meta data,
 * such as creation date, who created it, etc.
 * </p>
 * <p>
 * Comments support attachments, i.e. files that are stored as part of a comment, e.g. a preview image if the comment
 * is used to annotate an image.
 * </p>
 * <p>
 * Comments can also support reply to a comment
 * </p>
 */

public interface Comment<T extends CommentCollectionConfiguration> extends SocialCollectionComponent {

    /** JSON extension. */
    String JSON_EXTENSION = ".json";

    /** GET Selector. */
    String SOCO_SELECTOR = ".social";

    /** The comment resourcetype. */
    String COMMENT_RESOURCETYPE = "social/commons/components/hbs/comments/comment";

    /** The commentcollection resourcetype. */
    String COMMENTCOLLECTION_RESOURCETYPE = "social/commons/components/hbs/comments";

    /**
     * The comment node type.
     */
    public static final String NODE_TYPE = "cq:Comment";

    /**
     * The property name for the approval status.
     */
    public static final String PROP_APPROVED = "approved";

    /**
     * The property name for the spam flag.
     */
    public static final String PROP_SPAM = "isSpam";

    /**
     * The property name for use referrer as the @link getFriendlyUrl() value. This is used for pages with suffix in
     * its url.
     */
    String PROP_USE_REFERRER_URL = "useReferrer";

    /**
     * Request paramter and header name for Referrer Url.
     */
    String PROP_REFERER = "Referer";

    /**
     * Retrieve an attachment of this comment by its file name.
     * @param name The file name of the attachment.
     * @return A {@link ResourceID} representing the attachment, or <code>null</code> if no attachment with the given
     *         name exists.
     */
    Attachment getAttachment(String name);

    /**
     * Returns the author's identifying name that represents the author of this comment.
     * @return The User representing the author name.
     */
    User getAuthor();

    /**
     * Retrieves a map, an entry of which represents a file-name/attachment pair. The map contains all the attachments
     * of the comment.
     * @return A {@link Map} representing the available attachments, or an empty map if no attachments are present.
     */
    Map<String, Attachment> getAttachments();

    /**
     * The time stamp this comment was created.
     * @return A <code>Calendar</code> representing creation time.
     */
    Calendar getCreated();

    /**
     * Retrieves the message of this comment.
     * @return A <code>String</code> representing the message.
     */
    String getMessage();

    /**
     * @return true, if the comment is a top level comment; false, if the comment is a reply to another comment.
     */
    boolean isTopLevel();

    /**
     * Checks whether this comment is approved (in case comments are moderated).
     * @return <code>true</code> if comment is approved or unmoderated, <code>false</code> otherwise
     */
    public boolean isApproved();

    /**
     * @return <code>true</code> if the Edit action button should be displayed
     */
    boolean getCanEdit();

    /**
     * @return <code>true</code> if the Delete action button should be displayed
     */
    boolean getCanDelete();

    /**
     * Get this comment parent id.
     * @return this comment parent id.
     */
    String getParentId();

    /**
     * Check if the UGC language is the same as the Page language.
     * @return <code>true</code> if the ugc language is detected and enabled and different than the page language.
     */
    boolean getCanTranslate();

    /**
     * @return true if the current user is a moderator.
     */
    boolean isUserModerator();

    /**
     * @return a list of the reasons for which users have flagged this comment.
     */
    List<FlagReason> getFlagReasons();

    /**
     * @return true if the comment should be visible, given the current user, the state of the comment, and the state
     *         of the comment system.
     */
    @JsonProperty("isVisible")
    boolean isVisible();

    /**
     * @return true if the comment is closed, false otherwise.
     */
    @JsonProperty("isClosed")
    boolean isClosed();

    /**
     * Checks if comment is scheduled to be published later.
     * @return true, if is scheduled to be published later, false otherwise.
     */
    boolean isScheduled();

    /**
     * Checks if is draft.
     * @return true, if is draft
     */
    boolean isDraft();

    /**
     * Returns the scheduled publish date.
     * @return The scheduled publish date of the topic or <code>null</code> if not found.
     */
    Calendar getPublishDate();

    List<Tag> getTags();

    public interface Tag {
        String getTitle();

        String getTagId();
    }

    /**
     * @return voting component
     */
    SocialComponent getVotes();

    /**
     * @return the text the current user flagged this comment with; an empty String if it was flagged with no text; or
     *         null, if the current user did not flag this comment.
     */
    String getCurrentUserFlagText();

    /**
     * @return true if the reply button should be displayed.
     */
    boolean getCanReply();

    /**
     * This only available if the current user is a moderator
     * @return the comment moderator status
     */
    @JsonInclude(Include.NON_NULL)
    ModeratorStatus getModeratorStatus();

    /**
     * @return the {@link ModeratorActions}
     */
    ModeratorActions getModeratorActions();

    /**
     * Get the configuration for this comment
     * @return the configuration
     */
    T getConfiguration();

    public interface ModeratorStatus {
        /**
         * return true if the comment is flagged
         */
        @JsonProperty("isFlagged")
        boolean isFlagged();

        /**
         * return true if this comment is approved
         */
        @JsonProperty("isApproved")
        boolean isApproved();

        /**
         * return true if this comment is spam
         */
        @JsonProperty("isSpam")
        boolean isSpam();

        /**
         * return true if this comment is pending
         */
        @JsonProperty("isPending")
        boolean isPending();
    }

    public interface ModeratorActions {

        /**
         * @return true if the spam message should be displayed.
         */
        // getDisplaySpamMessage();

        /**
         * @return true if the flagged message should be displayed.
         */
        // getDisplayFlaggedMessage();

        // getDisplayModMessage(); !isApproved && isVisible

        /**
         * @return boolean true if the user can select a flag reason from a list.
         */
        boolean getUseFlagReasonList();

        /**
         * @return true if the close button should be displayed.
         */
        boolean getCanClose();

        /**
         * @return true if the Allow action button should be displayed.
         */
        boolean getCanAllow();

        /**
         * @return true if the Flag action button should be displayed.
         */
        boolean getCanFlag();

        /**
         * @return true if the Deny action button should be displayed.
         */
        boolean getCanDeny();

        /**
         * @return true, if the user can perform a move operation here
         */
        boolean getCanMove();

    }

    /**
     * Gets information about the pages for this collection. This can be used by the page block helper to easily
     * render various pagination UIs.
     * @return information about the pagination system
     */
    PageInfo getPageInfo();

    /**
     * @return boolean true if flag reasons are used, either from a list or custom.
     */
    boolean getUseFlagReasons();

    /**
     * @return boolean true if the current user has flagged this content.
     */
    @JsonProperty("isFlaggedByUser")
    boolean isFlaggedByUser();

    /**
     * @return a String that points to the location of the component that created this comment.
     */
    @JsonProperty("sourceComponentId")
    String getSourceComponentId();

    /**
     * @return a String that is the translation of the title of the discussion in page language.
     */
    String getTranslationTitle();

    /**
     * @return a String that is the translation of the comment in page language.
     */
    String getTranslationDescription();

    /**
     * @return a String that is the translation attribution( translator provider name ).
     */
    String getTranslationAttribution();

    /**
     * @return a String that is the display format of translation and the comment.
     */
    String getTranslationDisplay();

    /**
     * @return a String that is the referrer url for the page that contains this comment. This is only applicable if
     *         the {@link PROP_USE_REFERRER_URL} configuration is set to true.
     * @return
     */
    @JsonInclude(Include.NON_NULL)
    String getReferrerUrl();

}
