/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.cq.social.commons.comments.endpoints;

import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.ReferenceCardinality;
import org.apache.felix.scr.annotations.ReferencePolicy;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.jcr.api.SlingRepository;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.cq.social.commons.CollabUtil;
import com.adobe.cq.social.commons.CommentException;
import com.adobe.cq.social.commons.bundleactivator.Activator;
import com.adobe.cq.social.commons.comments.endpoints.impl.CommentPostOperation;
import com.adobe.cq.social.scf.OperationException;
import com.adobe.cq.social.scf.SocialComponent;
import com.adobe.cq.social.scf.SocialComponentFactory;
import com.adobe.cq.social.scf.SocialComponentFactoryManager;
import com.adobe.cq.social.scf.SocialOperationResult;
import com.adobe.cq.social.scf.core.operations.AbstractSocialOperation;
import com.adobe.cq.social.serviceusers.internal.ServiceUserWrapper;

/**
 * Base class of Comment Post Operation.
 */
@Component(metatype = false, componentAbstract = true)
public abstract class AbstractCommentOperation<T extends CommentOperations> extends AbstractSocialOperation {

    @Reference(cardinality = ReferenceCardinality.MANDATORY_UNARY, policy = ReferencePolicy.STATIC)
    protected SlingRepository repository;

    /**
     * We need this one because the one above believes its client is the bundle of the *concrete* component. e.g.
     * social-forums. The one here will know that it is used by social-commons and will thus obey our configurations
     * for service users tied to social-commons.
     */
    private SlingRepository repositoryBugFix;

    private ServiceUserWrapper serviceUserWrapper;

    /** Logger for this class. */
    private static final Logger LOG = LoggerFactory.getLogger(CommentPostOperation.class);

    private static final String UGC_WRITER = "ugc-writer";

    /** Social Component Factory Manager. */
    @Reference
    private SocialComponentFactoryManager componentFactoryManager;

    /**
     * Perform the operation and send the response using the provided session.
     * @param request the client request.
     * @param session the session to be used to perform the operation
     * @return SocialOperationResult the result of the operation.
     * @throws OperationException if there is an error occurs while performing the operation.
     */
    protected abstract SocialOperationResult performOperation(final SlingHttpServletRequest request,
        final Session session) throws OperationException;

    /**
     * Gets session from resource {@link Session}.
     * @param resource the resource for which to retrieve the session
     * @return The <code>Session</code> or <code>null</code> if the session couldn't be created.
     */
    protected Session getSessionFromResource(final Resource resource) {
        return resource.getResourceResolver().adaptTo(Session.class);
    }

    /**
     * Creates an administrative {@link Session}.
     * @return The <code>Session</code> or <code>null</code> if the session couldn't be created.
     */
    protected Session createAdminSession() {
        try {
            if (repositoryBugFix == null) {
                repositoryBugFix = Activator.getService(SlingRepository.class);
            }
            if (serviceUserWrapper == null) {
                serviceUserWrapper = Activator.getService(ServiceUserWrapper.class);
            }
            return serviceUserWrapper.loginService(repositoryBugFix, UGC_WRITER);
        } catch (final RepositoryException e) {
            LOG.error("error creating session: ", e);
        }
        return null;
    }

    /**
     * Logout an administrative {@link Session}.
     * @param serviceSession the administrative session to logout.
     */
    protected void closeAdminSession(final Session serviceSession) {
        if (serviceSession != null && serviceSession.isLive()) {
            serviceSession.logout();
        }
    }

    @Override
    protected SocialOperationResult performOperation(final SlingHttpServletRequest request) throws OperationException {
        Session session;
        Session serviceSession = null;
        if (CollabUtil.hasModeratePermissions(request.getResource())) {
            session = getSessionFromResource(request.getResource());
        } else {
            serviceSession = createAdminSession();
            session = serviceSession;
        }

        if (null == session) {
            throw new CommentException("session unavailable");
        }
        SocialOperationResult result;
        try {
            result = performOperation(request, session);
            return result;
        } finally {
            closeAdminSession(serviceSession);
        }
    }

    /**
     * Get the <code>SocialComponent</code> for the specified {@link Comment} and {@link SlingHttpServletRequest}.
     * @param comment the target comment
     * @param request the client request
     * @return the {@link SocialComponent}
     */
    protected SocialComponent getSocialComponentForComment(final Resource comment,
        final SlingHttpServletRequest request) {
        // resolving the resource again using the request session
        final Resource resource = request.getResourceResolver().getResource(comment.getPath());
        final SocialComponentFactory factory = componentFactoryManager.getSocialComponentFactory(resource);
        return (factory != null) ? factory.getSocialComponent(resource, request) : null;
    }

    /**
     * Get the {@link CommentOperations} service.
     * @return the CommentOperations service
     */
    protected abstract T getCommentOperationService();
}
