/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2015 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.commons.comments.endpoints;

import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.jackrabbit.commons.JcrUtils;
import org.apache.sling.api.resource.ModifiableValueMap;
import org.apache.sling.api.resource.PersistenceException;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ValueMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.cq.social.commons.comments.api.Comment;
import com.adobe.cq.social.scf.ClientUtilities;
import com.adobe.cq.social.ugcbase.SocialUtils;
import com.day.cq.commons.Externalizer;
import com.day.text.Text;

public class UploadOperationUtils {
    private static String UGC_ROOT_DIR = "/content/usergenerated";
    private static String TEMP_ROOT_DIR = "tmp";
    private static String SOCIAL_DIR = "social";
    private static String SKIP_DIRS[] = {"jcr:content"};
    private static String IMAGE_ELEMENT = "img";
    private static String SRC = "src";
    private static String SRC_ELEMENT = SRC + "=";
    private static String IMAGES_DIR = "images";
    private static String TEMP_DIR = UGC_ROOT_DIR + "/" + TEMP_ROOT_DIR + "/" + SOCIAL_DIR + "/" + IMAGES_DIR;

    private static final Logger LOG = LoggerFactory.getLogger(UploadOperationUtils.class);

    public static Node addAttachment(final Node root, final String name, final InputStream inputStream,
        final String mimeType) throws RepositoryException {
        /*
         * if (!socialUtils.checkPermission(resource.getResourceResolver(), resource.getPath(),
         * Session.ACTION_ADD_NODE)) { throw new
         * CommentException(String.format("Not allowed to add an attachment to the comment: %s", resource.getPath()));
         * }
         */
        // Some browsers supply full path
        final String fileName = FilenameUtils.getName(name);
        Node attachmentNode;
        Node contentNode;
        if (root.hasNode(name)) {
            attachmentNode = root.getNode(name);
            contentNode = attachmentNode.getNode("jcr:content");
        } else {
            attachmentNode = root.addNode(name, "nt:file");
            contentNode = attachmentNode.addNode("jcr:content", "nt:resource");
        }
        contentNode.setProperty("jcr:mimeType", mimeType);
        contentNode.setProperty("jcr:data", inputStream);
        contentNode.setProperty("jcr:lastModified", Calendar.getInstance());

        return attachmentNode;
    }

    public static Node createOrGetUploadResource(final Session session, final String path)
        throws PathNotFoundException, RepositoryException {
        Node ugc = session.getNode(UGC_ROOT_DIR);
        if (ugc != null) {
            Node tmp = JcrUtils.getOrAddFolder(ugc, TEMP_ROOT_DIR);
            String folders[] = path.split("/");
            Node parent = JcrUtils.getOrAddFolder(tmp, SOCIAL_DIR);
            parent = JcrUtils.getOrAddFolder(parent, IMAGES_DIR);
            for (int i = path.startsWith("/") ? 0 : 1; i < folders.length; i++) {
                if (!skipFolder(folders[i]))
                    parent = JcrUtils.getOrAddFolder(parent, folders[i]);
            }
            return parent;
        } else {
            return null;
        }
    }

    public static void moveImages(final Externalizer externalizer, final boolean author,
        final SocialUtils socialUtils, final Comment comment) throws PersistenceException {
        String message = comment.getMessage();
        if (message.contains(IMAGE_ELEMENT)) {
            final Resource resource = comment.getResource();
            com.adobe.cq.social.commons.Comment ocomment =
                resource.adaptTo(com.adobe.cq.social.commons.Comment.class);
            ResourceResolver resolver = comment.getResource().getResourceResolver();
            int startPos = 0;
            int index = -1;
            boolean updated = false;

            while ((index = StringUtils.indexOf(message, SRC_ELEMENT, startPos)) >= 0) {
                startPos = index + 1;
                // Get the image url
                int imgStartPos = StringUtils.indexOf(message, "\"", index);
                if (imgStartPos > 0) {
                    int imgEndPos = StringUtils.indexOf(message, "\"", imgStartPos + 1);
                    if (imgEndPos > 0) {
                        String imgUrl = StringUtils.substring(message, imgStartPos + 1, imgEndPos);
                        if (imgUrl.startsWith(TEMP_DIR)) {
                            String name = FilenameUtils.getName(imgUrl);
                            Resource imgResource = resolver.getResource(imgUrl);
                            if (imgResource != null && imgResource.hasChildren()) {
                                imgResource = imgResource.getChild("jcr:content");
                            }
                            if (imgResource != null) {
                                InputStream inputStream = imgResource.adaptTo(InputStream.class);
                                ValueMap valueMap = imgResource.adaptTo(ValueMap.class);
                                String mimeType = valueMap.get("jcr:mimeType", String.class);
                                Resource image = ocomment.addImage(name, inputStream, mimeType);
                                String externalUrl = externalizeLink(externalizer, author, resolver, image.getPath());
                                message = replace(message, externalUrl, imgStartPos, imgEndPos - imgStartPos);
                                // resolver.delete(imgResource);
                                updated = true;
                            }

                        }
                    }
                }
            }
            if (updated) {
                ModifiableValueMap mvm = resource.adaptTo(ModifiableValueMap.class);
                mvm.put("jcr:description", message);
                resolver.commit();
            }

        }
    }

    public static void moveImages(final ClientUtilities clientUtils, final Comment comment,
        final ResourceResolver serviceUserResolver) throws PersistenceException {
        String message = comment.getMessage();
        if (message.contains(IMAGE_ELEMENT)) {
            String commentPath = comment.getResource().getPath();
            final Resource resource = serviceUserResolver.getResource(commentPath);
            com.adobe.cq.social.commons.Comment ocomment =
                resource.adaptTo(com.adobe.cq.social.commons.Comment.class);
            int startPos = 0;
            int index = -1;
            boolean updated = false;
            Resource imgResource = null;
            List<Resource> deletedResources = new ArrayList<Resource>();
            Resource targetImgResource;
            while ((index = StringUtils.indexOf(message, SRC_ELEMENT, startPos)) >= 0) {
                startPos = index + 1;
                // Get the image url
                int imgStartPos = StringUtils.indexOf(message, "\"", index);
                if (imgStartPos > 0) {
                    int imgEndPos = StringUtils.indexOf(message, "\"", imgStartPos + 1);
                    if (imgEndPos > 0) {
                        String imgUrl = StringUtils.substring(message, imgStartPos + 1, imgEndPos);
                        imgUrl = Text.unescape(imgUrl);
                        if (imgUrl.startsWith(TEMP_DIR)) {
                            String name = FilenameUtils.getName(imgUrl);
                            imgResource = serviceUserResolver.getResource(imgUrl);
                            if (imgResource != null && imgResource.hasChildren()) {
                                targetImgResource = imgResource.getChild("jcr:content");
                            } else {
                                targetImgResource = imgResource;
                            }
                            if (targetImgResource != null) {
                                InputStream inputStream = targetImgResource.adaptTo(InputStream.class);
                                ValueMap valueMap = targetImgResource.adaptTo(ValueMap.class);
                                String mimeType = valueMap.get("jcr:mimeType", String.class);
                                Resource image = ocomment.addImage(name, inputStream, mimeType);
                                if (image != null) {
                                    deletedResources.add(imgResource);
                                    String externalUrl = encode(image.getPath(), clientUtils);
                                    message = replace(message, externalUrl, imgStartPos, imgEndPos - imgStartPos);
                                    updated = true;
                                }
                            }

                        }
                    }
                }
            }
            if (updated) {
                ModifiableValueMap mvm = resource.adaptTo(ModifiableValueMap.class);
                mvm.put("jcr:description", message);
                for (Resource deletedResource : deletedResources) {
                    Resource parentResource = deletedResource.getParent();
                    serviceUserResolver.delete(deletedResource);
                }
                serviceUserResolver.commit();
            }

        }

    }

    private static String encode(final String str, final ClientUtilities clientUtils) {
        String result = clientUtils.externalLink(str, false);
        // TODO: We can't really externalize the link since the clientUtils that the OperationExtension
        // doesn't have access to the HttpRequest..
        return Text.escapePath(str);
    }

    private static String replace(String text, String replacement, int startIndex, int length) {

        int replLength = replacement.length();
        int increase = replacement.length() - length;
        increase = (increase < 0 ? 0 : increase);
        increase *= 16;
        StringBuilder buf = new StringBuilder(text.length() + increase);
        buf = buf.append(text, 0, startIndex + 1).append(replacement).append(text.substring(startIndex + length));

        return buf.toString();
    }

    private static String externalizeLink(final Externalizer externalizer, final boolean author,
        final ResourceResolver resolver, final String url) {
        String externalUrl = url;
        if (externalizer != null) {
            if (author) {
                externalUrl = externalizer.authorLink(resolver, externalUrl);
            } else {
                externalUrl = externalizer.publishLink(resolver, externalUrl);
            }
        }
        return getRootBasedURL(url);
    }

    private static String getRootBasedURL(String url) {
        try {
            URL startURL = new URL(url);
            final String HOST_CHECK_HEADER = "origin";
            // Only mess with stuff if it's http[s]
            if (StringUtils.startsWithIgnoreCase(startURL.getProtocol(), "http")) {
                return StringUtils.removeStart(url.toString(), startURL.getProtocol() + "://" + startURL.getHost()
                        + ((startURL.getPort() == -1) ? "" : ":" + startURL.getPort()));

            }
        } catch (MalformedURLException e) {
            // Not sure if we can even get here because that means the externalizer generated a link without http?
        }
        return url;
    }

    private static boolean skipFolder(final String name) {
        for (int i = 0; i < SKIP_DIRS.length; i++) {
            if (name.equals(SKIP_DIRS[i]))
                return true;
        }
        return false;
    }

}