/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2015 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.cq.social.commons.emailreply;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.observation.Event;
import javax.jcr.observation.EventIterator;
import javax.jcr.observation.EventListener;
import javax.jcr.observation.ObservationManager;
import javax.mail.Address;
import javax.mail.BodyPart;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.internet.MimeMessage;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.mail.Email;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Deactivate;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.ReferenceCardinality;
import org.apache.felix.scr.annotations.ReferencePolicy;
import org.apache.jackrabbit.api.observation.JackrabbitEvent;
import org.apache.sling.api.resource.LoginException;
import org.apache.sling.api.resource.PersistenceException;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceResolverFactory;
import org.apache.sling.api.resource.ValueMap;

import org.osgi.service.component.ComponentContext;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.cq.social.commons.FileDataSource;
import com.adobe.cq.social.commons.bundleactivator.Activator;
import com.adobe.cq.social.commons.emailreply.internal.EmailReplyConfiguration;
import com.adobe.cq.social.serviceusers.internal.ServiceUserWrapper;

import com.day.cq.commons.jcr.JcrConstants;
import com.day.cq.mailer.MailService;

/**
 * This abstract class provides functionality to listen to email node creation event so that the implementer of this
 * class can process it as per its requirements.
 * <p/>
 * <p/>
 * CQ Feed Importer services are used to save mail to a repository location, path of which needs to be configured in
 * "email.import.path" property. JCR Observation mechanism detects node creation at this location from which emails
 * are read, parsed and processed as per the requirement of the implementer of this class. After processing, these
 * email nodes are deleted from repository.
 * <p/>
 * <p/>
 * Email content might contain quoted data along with actual post content. {@link EmailClientProvider} services are
 * used to parse and extract relevant information. Suitable {@link EmailClientProvider} service is selected based on
 * mail content.
 * <p/>
 * <p/>
 * Post is parsed from "Subject". If not found, post is parsed from "To" header of the inbound mail. Format of the
 * header is emailID+Comment Path@Domain Name.
 * <p/>
 * <p/>
 * Author is detected from "From" header of inbound mail. First email address is considered as author email id.
 * <p/>
 * Two properties are associated with email Nodes which indicate processing status namely "processingStatus": values
 * can be {processing, success, failure} and "processingStatusMsg": set when error occurred while processing email,
 * stores error message and sends failure email to the sender.
 */
@Component(componentAbstract = true)
public abstract class AbstractEmailEventListener implements EventListener {

    /**
     * Property for email client that parsed the reply email.
     */
    protected static final String EMAIL_CLIENT_PROVIDER_PROPERTY = "email.parsingEmailClient";

    /**
     * Property tracker id of comment object resource to which post parsed from email is to be added.
     */
    protected static final String TRACKER_ID_PROPERTY = "commentObjectUniqueId_s";

    /**
     * Property which indicates processing status of imported email.
     */
    private static final String PROCESSING_STATUS_PROPERTY = "processingStatus";

    /**
     * Property which indicates processing status message of imported email.
     */
    private static final String PROCESSING_STATUS_MSG_PROP = "processingStatusMsg";

    /**
     * Alias for oauthservice service user.
     */
    private static final String USER_READER = "user-reader";

    /**
     * Alias for msm-service service user.
     */
    private static final String MSM_SERVICE = "msm-service";

    /**
     * Alias for communities-ugc-writer service user.
     */
    private static final String UGC_WRITER = "ugc-writer";

    private static final Logger LOG = LoggerFactory.getLogger(AbstractEmailEventListener.class);

    @Reference(cardinality = ReferenceCardinality.MANDATORY_MULTIPLE, bind = "bindEmailClientProvider",
            unbind = "unbindEmailClientProvider", referenceInterface = EmailClientProvider.class,
            policy = ReferencePolicy.DYNAMIC)
    private static final List<EmailClientProvider> EMAIL_CLIENT_PROVIDERS = Collections
        .synchronizedList(new ArrayList<EmailClientProvider>());

    @Reference
    private ServiceUserWrapper serviceUserWrapper;

    @Reference
    private EmailReplyConfiguration emailReplyConfiguration;

    @Reference
    private MailService mailService;

    @Reference
    private FileDataSourceFactory fileDataSourceFactory;

    @Reference
    private CommentEmailBuilder commentEmailBuilder;

    /**
     * Executes email import events asynchronously.
     */
    private ExecutorService eventExecutorService;

    /**
     * The resource resolver used to obtain access for the reading to the users and usergenerated content directories.
     */
    private ResourceResolver userReaderResourceResolver;

    /**
     * The resource resolver used to obtain access to the email templates directory.
     */
    private ResourceResolver utilityReaderResourceResolver;

    /**
     * The resource resolver used to obtain access for writing to the usergenerated content directory.
     */
    private ResourceResolver ugcWriterResourceResolver;

    private ObservationManager observationManager;
    private Pattern pattern;

    /**
     * Activation.
     * @param context {@link ComponentContext}
     * @throws RepositoryException thrown by {@code configure()} method
     * @throws LoginException thrown by {@code configure()} method
     */
    protected void activate(final ComponentContext context) throws RepositoryException, LoginException {
        LOG.info("Activating {}.", getClass().getName());
        configure();
    }

    /**
     * Configuration.
     * @throws RepositoryException thrown by getting {@link ObservationManager}
     * @throws LoginException thrown by getting {@link ResourceResolver}
     */
    private void configure() throws RepositoryException, LoginException {
        final ResourceResolverFactory resourceResolverFactory = Activator.getService(ResourceResolverFactory.class);
        userReaderResourceResolver =
            serviceUserWrapper.getServiceResourceResolver(resourceResolverFactory,
                Collections.singletonMap(ResourceResolverFactory.SUBSERVICE, (Object) USER_READER));
        utilityReaderResourceResolver =
            serviceUserWrapper.getServiceResourceResolver(resourceResolverFactory,
                Collections.singletonMap(ResourceResolverFactory.SUBSERVICE, (Object) MSM_SERVICE));
        ugcWriterResourceResolver =
            serviceUserWrapper.getServiceResourceResolver(resourceResolverFactory,
                Collections.singletonMap(ResourceResolverFactory.SUBSERVICE, (Object) UGC_WRITER));
        if (ugcWriterResourceResolver != null) {
            observationManager =
                ugcWriterResourceResolver.adaptTo(Session.class).getWorkspace().getObservationManager();
        } else {
            throw new EmailReplyException("Unable to get observation manager. No resource resolver.");
        }
        eventExecutorService = Executors.newSingleThreadExecutor(new ThreadFactory() {
            @Override
            public Thread newThread(final Runnable r) {
                final Thread t = Executors.defaultThreadFactory().newThread(r);
                t.setDaemon(true);
                return t;
            }
        });
        pattern =
            Pattern.compile("(\\[" + emailReplyConfiguration.getTrackerIdPrefixInSubject() + ")([a-zA-Z0-9]+)(\\])");
    }

    /**
     * @see Reference#bind().
     * @param boundProvider registered {@link EmailClientProvider}
     */
    protected void bindEmailClientProvider(final EmailClientProvider boundProvider) {
        EMAIL_CLIENT_PROVIDERS.add(boundProvider);
        Collections.sort(EMAIL_CLIENT_PROVIDERS, new Comparator<EmailClientProvider>() {
            @Override
            public int compare(final EmailClientProvider p1, final EmailClientProvider p2) {
                final Integer priority1 = p1.getPriorityOrder();
                final Integer priority2 = p2.getPriorityOrder();
                return priority1.compareTo(priority2);
            }
        });
        LOG.info("Bound email client provider: {}", boundProvider.getClass().getSimpleName());
    }

    /**
     * @see Reference#unbind().
     * @param unboundProvider unregistered {@link EmailClientProvider}
     */
    protected void unbindEmailClientProvider(final EmailClientProvider unboundProvider) {
        EMAIL_CLIENT_PROVIDERS.remove(unboundProvider);
        LOG.debug("Unbound email client provider: {}", unboundProvider.getClass().getSimpleName());
    }

    /**
     * Get the {@link ResourceResolver} instance for oauthservice service user.
     * @return the instance of {@link ResourceResolver}
     */
    protected ResourceResolver getUserReaderResourceResolver() {
        return userReaderResourceResolver;
    }

    /**
     * Get the {@link ResourceResolver} instance for communities-ugc-writer.
     * @return the instance of {@link ResourceResolver}
     */
    protected ResourceResolver getUgcWriterResourceResolver() {
        return ugcWriterResourceResolver;
    }

    /**
     * Get the {@link ObservationManager} instance.
     * @return the instance of {@link ObservationManager}
     */
    protected ObservationManager getObservationManager() {
        return observationManager;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void onEvent(final EventIterator events) {
        while (events.hasNext()) {
            final Event event = events.nextEvent();
            if (event instanceof JackrabbitEvent && ((JackrabbitEvent) event).isExternal()) {
                continue;
            }
            eventExecutorService.execute(new Runnable() {
                @Override
                public void run() {
                    String nodePath = null;
                    Resource emailResource = null;
                    Resource resource;
                    EmailContent emailContent = null;
                    try {
                        nodePath = event.getPath();
                        LOG.debug("Email received at path: [{}]", nodePath);
                        if (!nodePath.contains("/Job") && nodePath.endsWith("/message/jcr:content")) {
                            resource = ugcWriterResourceResolver.getResource(nodePath);
                            emailResource = resource.getParent().getParent();
                            setProcessingStatus(emailResource, "processing", null);
                            emailContent = getEmailContent(resource);
                            processEvent(resource, emailContent);
                            LOG.info("Deleting node: [{}]", emailResource.getPath());
                            ugcWriterResourceResolver.delete(emailResource);
                            ugcWriterResourceResolver.commit();
                        }
                    } catch (final RepositoryException e) {
                        LOG.warn("Error while processing event", e);
                    } catch (final PersistenceException e) {
                        LOG.warn("Unable to delete processed email at path: [{}]", nodePath, e);
                    } catch (final EmailReplyException e) {
                        LOG.warn("Error while processing email saved at path: [{}]", nodePath, e);
                        setProcessingStatus(emailResource, "failure", e.getMessage());
                        sendFailureEmailReply(emailContent);
                    }
                }
            });
        }
    }

    /**
     * Parses reply email data and returns information needed to create a post including list of attachments.
     * Mandatory parameters to be parsed from mail are email message, tracker id and author id. Currently author email
     * id is used as author id for author profile identification.
     * @param resource the email resource that has been created
     * @return mailContent object that contains properties of mail containing information about email content, comment
     *         object id, author id and parsing status message and the list of attachments in the email
     */
    public EmailContent getEmailContent(final Resource resource) {
        final ValueMap valueMap = resource.adaptTo(ValueMap.class);
        final InputStream emailStream = (InputStream) valueMap.get(JcrConstants.JCR_DATA);
        final Properties properties = new Properties();
        final List<FileDataSource> attachmentList = new ArrayList<FileDataSource>();
        try {
            final MimeMessage email = getEmailFromStream(emailStream);
            final EmailClientProvider emailClientProvider = getEmailClientProvider(email);
            final EmailContent emailContent = new EmailContent() {
                @Override
                public Properties getProperties() {
                    return properties;
                }

                @Override
                public List<FileDataSource> getAttachments() {
                    return attachmentList;
                }
            };

            String emailClientProviderName;
            if (emailClientProvider == null) {
                throw new EmailReplyException("No suitable email client provider found.");
            } else {
                emailClientProviderName = emailClientProvider.getClass().getName();
                LOG.debug("Contents parsed by {}", emailClientProviderName);
            }

            final String trackerId = getTrackerIdFromMail(email);
            final Properties emailProperties = emailClientProvider.getMailProperties(email);

            properties.setProperty(EmailClientProvider.EMAIL_MESSAGE_PROPERTY,
                emailProperties.getProperty(EmailClientProvider.EMAIL_MESSAGE_PROPERTY, ""));
            properties.setProperty(EmailClientProvider.EMAIL_AUTHOR_PROPERTY,
                emailProperties.getProperty(EmailClientProvider.EMAIL_AUTHOR_PROPERTY, ""));
            properties.setProperty(EmailClientProvider.EMAIL_SUBJECT_PROPERTY,
                emailProperties.getProperty(EmailClientProvider.EMAIL_SUBJECT_PROPERTY, ""));
            properties.setProperty(TRACKER_ID_PROPERTY, StringUtils.defaultIfEmpty(trackerId, ""));
            properties.setProperty(EMAIL_CLIENT_PROVIDER_PROPERTY,
                StringUtils.defaultIfEmpty(emailClientProviderName, ""));

            handleAttachments(attachmentList, email);

            return emailContent;
        } catch (final MessagingException e) {
            throw new EmailReplyException("Unable to construct MIME message from input stream", e);
        } finally {
            IOUtils.closeQuietly(emailStream);
        }
    }

    /**
     * Constructs a MimeMessage by reading and parsing the data from the mail {@link InputStream}.
     * @param mailFileStream input stream representing mail
     * @return MimeMessage object representing that mail
     * @throws MessagingException
     */
    private MimeMessage getEmailFromStream(final InputStream mailFileStream) throws MessagingException {
        final Properties properties = System.getProperties();
        final javax.mail.Session mailSession = javax.mail.Session.getDefaultInstance(properties, null);
        return new MimeMessage(mailSession, mailFileStream);
    }

    /**
     * Scans through active {@link EmailClientProvider} services to identify appropriate service for specified email.
     * This will be used for extracting necessary data from email.
     * @param mail mail message to be parsed.
     * @return suitable email client provider service. {@code null} if none is found.
     */
    private EmailClientProvider getEmailClientProvider(final MimeMessage mail) {
        for (final EmailClientProvider emailClientProvider : EMAIL_CLIENT_PROVIDERS) {
            final boolean mailAccepted = emailClientProvider.acceptsMail(mail);
            LOG.debug("Going through EmailClientProvider {} to see if this accepts mail {}.", emailClientProvider
                .getClass().getName(), mailAccepted);
            if (mailAccepted) {
                return emailClientProvider;
            }
        }
        return null;
    }

    /**
     * Returns tracker id corresponding to comment object id for which post will be added. Tracker id is in subject,
     * by default. When a mail notification is sent for new post, subject is appended by comment object id hash
     * (format: emailSubject [post#trackerId]) trackerId is a hexadecimal string and can be extracted using group 2 of
     * regex ([post#)([a-zA-Z0-9]+)(]). If tracker id was not found in subject, it might be present in reply-to header
     * with format: emailID+trackerId@domain. Returns "" if tracker id cannot be parsed.
     * @param emailMessage mail message used for parsing.
     * @return tracker id or {@code ""}.
     * @throws MessagingException
     */
    private String getTrackerIdFromMail(final MimeMessage emailMessage) throws MessagingException {
        String trackerId = "";
        final String subject = emailMessage.getSubject();
        if (subject != null) {
            final Matcher m = pattern.matcher(subject);
            if (m.find()) {
                LOG.debug("Found tracker id in subject: {}", trackerId);
                return m.group(2);
            }
        }
        final Address[] recipients = emailMessage.getRecipients(Message.RecipientType.TO);
        if (recipients != null && recipients.length > 0) {
            String recipientAddress = recipients[0].toString();
            if (recipientAddress.contains("<") && recipientAddress.contains(">")) {
                recipientAddress = recipientAddress.replaceAll("[<>]", "");
            }
            final int replyToDelimiterIndex = recipientAddress.indexOf(emailReplyConfiguration.getReplyToDelimiter());
            if (replyToDelimiterIndex != -1) {
                trackerId = recipientAddress.substring(replyToDelimiterIndex + 1, recipientAddress.indexOf("@"));
                LOG.debug("Found tracker id in recipient address [{}]", trackerId);
            } else {
                LOG.error("Recipient email address has no configured reply-to delimiter.");
            }
        } else {
            LOG.error("Unable to track recipient email address.");
        }
        return trackerId;
    }

    /**
     * Handle attachments list to which files will be added.
     * @param attachments attachments list to which files will be added
     * @param mail MimeMessage that is needed to be parsed
     */
    private void handleAttachments(final List<FileDataSource> attachments, final MimeMessage mail) {
        try {
            if (mail == null) {
                LOG.error("Unable to handle attachments list. No MIME message.");
                return;
            }
            final Object mailContent = mail.getContent();
            if (mailContent instanceof Multipart) {
                final Multipart multipart = (Multipart) mailContent;
                handleMultipart(attachments, multipart);
            }
        } catch (final MessagingException e) {
            LOG.error("Unable to construct MIME message from input stream", e);
        } catch (final IOException e) {
            LOG.error("Unable to handle attachments list in the email", e);
        }
    }

    /**
     * Attach files from the {@link Multipart} of email to the attachments list.
     * @param attachments attachments list to which files will be added
     * @param multipart content parsed from the email
     * @throws MessagingException
     * @throws IOException
     */
    private void handleMultipart(final List<FileDataSource> attachments, final Multipart multipart)
        throws MessagingException, IOException {
        final int count = multipart.getCount();
        for (int i = 0; i < count; i++) {
            final BodyPart bodypart = multipart.getBodyPart(i);
            final Object content = bodypart.getContent();
            if (bodypart.getFileName() != null) {
                if (content instanceof String || content instanceof InputStream) {
                    final FileDataSource attachment = fileDataSourceFactory.getFileDataSource(bodypart);
                    attachments.add(attachment);
                } else {
                    LOG.error("Unable to attach file with name " + bodypart.getFileName());
                }
            } else if (content instanceof Multipart) {
                handleMultipart(attachments, (Multipart) content);
            }
        }
    }

    /**
     * Sends failure notification email to the sender of the email if the email fails to be processed.
     * @param emailContent the parsed content of the mail which failed to perform the desired task.
     */
    public void sendFailureEmailReply(final EmailContent emailContent) {
        final Email failureEmail =
            commentEmailBuilder.buildFailure(userReaderResourceResolver, utilityReaderResourceResolver, emailContent);
        if (failureEmail != null) {
            LOG.info("Sending failure notification email.");
            mailService.send(failureEmail);
        } else {
            LOG.error("Unable to send failure notification email. Email hasn't been built.");
        }
    }

    /**
     * Updates processing status of email nodes.
     * @param emailResource email resource
     * @param processingStatus processing status
     * @param statusMsg additional processing message
     */
    protected void setProcessingStatus(final Resource emailResource, final String processingStatus,
        final String statusMsg) {
        if (emailResource != null) {
            try {
                final Node emailNode = emailResource.adaptTo(Node.class);
                if (processingStatus != null) {
                    emailNode.setProperty(PROCESSING_STATUS_PROPERTY, processingStatus);
                }
                if (statusMsg != null) {
                    emailNode.setProperty(PROCESSING_STATUS_MSG_PROP, statusMsg);
                }
            } catch (final RepositoryException e) {
                LOG.error("Error while saving setting processing status", e);
            }
        }
    }

    /**
     * @param componentContext {@link ComponentContext}
     */
    @Deactivate
    protected void deactivate(final ComponentContext componentContext) {
        if (eventExecutorService != null) {
            eventExecutorService.shutdown();
        }
        if (userReaderResourceResolver != null) {
            userReaderResourceResolver.close();
        }
        if (utilityReaderResourceResolver != null) {
            utilityReaderResourceResolver.close();
        }
        if (ugcWriterResourceResolver != null) {
            ugcWriterResourceResolver.close();
        }
    }

    /**
     * Processes the parsed email content.
     * @param resource The resource of email node
     * @param emailContent contains the email properties, attachments and email client used for parsing the email
     */
    protected abstract void processEvent(final Resource resource, final EmailContent emailContent);
}
