/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2015 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.cq.social.commons.emailreply;

import java.io.IOException;
import java.util.Properties;

import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;

/**
 * The <code>EmailClientProvider</code> specifies the contract for different email client service providers.
 * <p/>
 * EmailClientProvider services are used to parse relevant information (like post, post author, etc) from mail
 * message. Whether the EmailClientProvider can parse given mailMessage can be checked using acceptsMail() method. For
 * EmailClientProvider which parses mailMessage for creation of post, mandatory set of properties which needs to be
 * parsed are <code>email.message</code> and <code>email.author.id</code>.
 * <p/>
 * EmailClientProvider services has been intended to provide dynamic support for various email clients like gmail,
 * outlook which use different patterns of quoted text in their reply email.
 * <p/>
 * Abstract implementation is provided by <code>AbstractEmailClientProvider.java</code>
 */
public interface EmailClientProvider {

    /**
     * Property for message contents parsed from email body.
     */
    String EMAIL_MESSAGE_PROPERTY = "email.message";

    /**
     * Property for author of email, i.e. "From" address.
     */
    String EMAIL_AUTHOR_PROPERTY = "email.author";

    /**
     * Property for subject of email.
     */
    String EMAIL_SUBJECT_PROPERTY = "email.subject";

    /**
     * Returns true if mailMessage can be parsed using this EmailClientProvider else false. In context of post from
     * inbound mail functionality, this method can check whether quoted text pattern in reply mail be identified and
     * removed by this EmailClientProvider to get post message.
     * @param mailMessage mail to be parsed
     * @return true if parsing is possible else false
     */
    boolean acceptsMail(final MimeMessage mailMessage);

    /**
     * Returns a set of mail properties parsed from mailMessage. In context of post from inbound mail functionality,
     * mandatory parameters to be parsed are <code>email.message</code> and <code>email.author.id</code>.
     * @param mailMessage mail whose properties is to be parsed
     * @return mail properties parsed from mailMessage. Empty property if no relevant property can be parsed
     */
    Properties getMailProperties(final MimeMessage mailMessage);

    /**
     * Returns mailMessage content. If content is instance of String, it is returned. In case content is instance of
     * <code>Multipart </code>, first part with content type "text/plain" is returned. Quoted text, in case of reply
     * emails, is not pruned from content.
     * <p/>
     * <p/>
     * Can be extended by implementations for more complex parsing of mail content.
     * @param mailMessage mail whose content is to be read
     * @return mail content. Empty String if content is null
     * @throws IOException thrown by MimeMessage DataHandler
     * @throws MessagingException thrown by MimeMessage DataHandler
     */
    String getEmailContent(final MimeMessage mailMessage) throws IOException, MessagingException;

    /**
     * Returns the priority rank deciding the order in which this email client provider will be used to parse reply
     * emails.
     * @return priority order
     */
    int getPriorityOrder();
}
