/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.cq.social.commons.events;

import java.util.HashMap;

import org.apache.commons.lang3.StringUtils;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceUtil;
import org.apache.sling.api.resource.ValueMap;
import org.osgi.service.event.Event;

import com.adobe.cq.social.commons.comments.api.Comment;
import com.adobe.cq.social.commons.events.CommentEvent.CommentActions;
import com.adobe.cq.social.scf.SocialComponent;
import com.adobe.cq.social.scf.core.SocialEvent;
import com.adobe.granite.activitystreams.ObjectTypes;
import com.adobe.granite.activitystreams.Verbs;
import com.day.text.Text;

/**
 * Defines events that are triggered in a comment system.
 */
public class CommentEvent extends SocialEvent<CommentActions> {

    private static final long serialVersionUID = 1L;

    /**
     * The event topic suffix for comment events.
     */
    public static final String COMMENT_TOPIC = "comment";

    private static final String COMMENT_SYSTEM_PATH = "COMMENT_SYSTEM_PATH";

    /**
     * List of available comment actions that can trigger a comment event.
     */
    public static enum CommentActions implements SocialEvent.SocialActions {
        CREATED, REPLIED, EDITED, DELETED;

        @Override
        public String getVerb() {
            switch (this) {
                case CREATED:
                    return Verbs.POST;
                case REPLIED:
                    return Verbs.ADD;
                case EDITED:
                    return Verbs.UPDATE;
                case DELETED:
                    return Verbs.DELETE;
                default:
                    throw new IllegalArgumentException();
            }
        }

    }

    /**
     * @param commentPath - the path to the comment node on which the event was triggered
     * @param commentSystemPath - the path to the comment system that the comment belongs to.
     * @param userId - the user id of the user who triggered the action.
     * @param action - the action that was taken to trigger the event.
     */
    public CommentEvent(final Comment comment, final String userId, final CommentActions action) {
        super(COMMENT_TOPIC, comment.getResource().getPath(), userId, action, new BaseEventObject(
            getObjectName(comment.getResource()), comment.getResource().getPath(), ObjectTypes.COMMENT),
            new BaseEventObject(getTargetname(comment), (comment.getParentComponent() != null) ? comment
                .getParentComponent().getResource().getPath() : comment.getSourceComponentId(),
                ObjectTypes.COLLECTION), new HashMap<String, Object>(1) {
                private static final long serialVersionUID = 1L;
                {
                    this.put(COMMENT_SYSTEM_PATH, (comment.getParentComponent() != null) ? comment
                        .getParentComponent().getResource().getPath() : comment.getSourceComponentId());
                }
            });
    }

    /**
     * Compute the display name of the object
     * @param resource
     * @return
     */
    protected static String getObjectName(final Resource resource) {
        return "a " + Text.getName(resource.getResourceType());
    }

    /**
     * Compute the display name of the target
     * @param comment
     * @return
     */
    protected static String getTargetname(final Comment comment) {
        String title = null;
        SocialComponent component = comment.getParentComponent();
        if (component != null) {
            ValueMap vm = ResourceUtil.getValueMap(component.getResource());
            title = vm.get("jcr:title", String.class);
            if (StringUtils.isEmpty(title)) {
                title = Text.getName(component.getId().getResourceIdentifier());
                if (component.getResource().isResourceType(Comment.COMMENT_RESOURCETYPE)) {
                    title = StringUtils.substringAfterLast(title, "-");
                }
            }
        }
        if (StringUtils.isEmpty(title)) {
            title = Text.getName(comment.getSourceComponentId());
        }
        return title;
    }

    /**
     * This constructor should be used to create a Comment Event given an OSGi Event.
     * @param event an OSGi event that is known to be a comment event.
     */
    private CommentEvent(final Event event) {
        super(event);
    }

    /**
     * @return the path to the comment system of the comment on which this event was triggered.
     */
    public String getCommentSystemPath() {
        return (String) this.getProperty(COMMENT_SYSTEM_PATH);
    }

    /**
     * Retrieves the comment event from the given generic {@link Event}.
     * @param event The event.
     * @return The comment event or <code>null</code> if the given event is not a comment event.
     */
    public static CommentEvent fromEvent(final Event event) {
        if ((com.adobe.cq.social.scf.core.SocialEvent.SOCIAL_EVENT_TOPIC_PREFIX + COMMENT_TOPIC).equals(event
            .getTopic())) {
            return new CommentEvent(event);
        }
        return null;
    }

}
