/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.cq.social.commons.moderation.api;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.servlet.http.HttpServletResponse;

import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.ReferenceCardinality;
import org.apache.felix.scr.annotations.ReferencePolicy;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.resource.LoginException;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceResolverFactory;
import org.apache.sling.jcr.api.SlingRepository;
import org.apache.sling.jcr.resource.JcrResourceConstants;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.cq.social.commons.bundleactivator.Activator;
import com.adobe.cq.social.scf.OperationException;
import com.adobe.cq.social.scf.SocialComponent;
import com.adobe.cq.social.scf.SocialComponentFactory;
import com.adobe.cq.social.scf.SocialComponentFactoryManager;
import com.adobe.cq.social.scf.SocialOperationResult;
import com.adobe.cq.social.scf.core.operations.AbstractSocialOperation;
import com.adobe.cq.social.serviceusers.internal.ServiceUserWrapper;
import com.adobe.cq.social.ugcbase.CollabUser;
import com.adobe.cq.social.ugcbase.SocialUtils;

/**
 * Base class of moderation operations
 */
@Component(metatype = false, componentAbstract = true)
public abstract class AbstractModerationOperation extends AbstractSocialOperation {

    public final String BULK_PARAM = ":items";

    /**
     * Implementors of this ModerationOperations interface perform the actual underlying operations
     */
    @Reference
    protected ModerationOperations moderationOps;

    @Reference(cardinality = ReferenceCardinality.MANDATORY_UNARY, policy = ReferencePolicy.STATIC)
    protected SlingRepository repository;

    /** Social Component Factory Manager. */
    @Reference
    protected SocialComponentFactoryManager componentFactoryManager;

    /** SocialUtils. */
    @Reference
    protected SocialUtils socialUtils;

    /**
     * We need this one so that we can use a SlingRepository which will lookup service users under the
     * cq-social-commons bundle name, not the bundle name of the concrete implementation.
     */
    private SlingRepository repositoryBugFix;

    private ServiceUserWrapper serviceUserWrapper;

    /** Logger for this class. */
    private static final Logger log = LoggerFactory.getLogger(AbstractModerationOperation.class);

    /**
     * The name of the property that contains defines whether the comment should be reverse-replicated.
     */
    // TODO This constant and some others should be moved somewhere more general, usable, and meaningful. ?
    public static final String PROPERTY_REVERSE_REPLICATE = "reverseReplicate";

    private static final String UGC_WRITER = "ugc-writer";

    /**
     * Resource Resolver used for the request.
     */
    @Reference(cardinality = ReferenceCardinality.MANDATORY_UNARY, policy = ReferencePolicy.STATIC)
    protected ResourceResolverFactory resourceResolverFactory;

    /**
     * Perform the operation and send the response using the provided session
     * @param request
     * @throws OperationException
     */
    public abstract String getOperationName();

    /**
     * Perform the operation and send the response using the provided session
     * @param request
     * @throws OperationException
     */
    @Override
    public abstract SocialOperationResult performOperation(final SlingHttpServletRequest request)
        throws OperationException;

    /**
     * Get the session from a resource {@link Session}.
     * @param resource the resource for which to retrieve the session
     * @return The <code>Session</code> or <code>null</code> if the session couldn't be created.
     */
    protected Session getSession(final Resource resource) {
        return resource.getResourceResolver().adaptTo(Session.class);
    }

    /**
     * Get the session from request {@link Session}.
     * @param request the original servlet request
     * @return The <code>Session</code> or <code>null</code> if the session couldn't be created.
     */
    protected Session getSession(final SlingHttpServletRequest request) {
        return getSession(request.getResource());
    }

    /**
     * Creates an administrative {@link Session}.
     * @return The <code>Session</code> or <code>null</code> if the session couldn't be created.
     */
    protected Session createAdminSession() {
        try {
            if (repositoryBugFix == null) {
                repositoryBugFix = Activator.getService(SlingRepository.class);
            }
            if (serviceUserWrapper == null) {
                serviceUserWrapper = Activator.getService(ServiceUserWrapper.class);
            }
            return serviceUserWrapper.loginService(repositoryBugFix, UGC_WRITER);
        } catch (final RepositoryException e) {
            log.error("error creating session: ", e);
        }
        return null;
    }

    /**
     * Logout an administrative {@link Session}.
     * @param adminSession the administrative session to logout.
     */
    protected void closeAdminSession(final Session adminSession) {
        if (adminSession != null && adminSession.isLive()) {
            adminSession.logout();
        }
    }

    /**
     * Return the resolved resource from the session.
     * @param request the servlet request
     * @param session the session associated with the resource
     * @throws OperationException
     */
    public Resource getSessionResource(final SlingHttpServletRequest request, final Session session)
        throws OperationException {
        final Resource requestResource = request.getResource();

        final Map<String, Object> sessionMap = new HashMap<String, Object>();

        // TODO Replace the constant below with something non-jcr
        sessionMap.put(JcrResourceConstants.AUTHENTICATION_INFO_SESSION, session);
        ResourceResolver resolver;
        try {
            resolver = resourceResolverFactory.getResourceResolver(sessionMap);
        } catch (final LoginException e) {
            throw new OperationException("Unable to get session resource from session.  ", e,
                HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
        }

        return resolver.getResource(requestResource.getPath());
    }

    protected SocialOperationResult getSocialOperationResult(final SlingHttpServletRequest request, final String path) {

        SocialComponent socialComponent = null;
        // getting the resource from resolver again to avoid stale data issues in the slingRequest resource
        final Resource resource = request.getResourceResolver().getResource(request.getResource().getPath());
        final SocialComponentFactory factory = componentFactoryManager.getSocialComponentFactory(resource);
        if (factory != null) {
            socialComponent = factory.getSocialComponent(resource, request);
        }

        return new SocialOperationResult(socialComponent, HttpServletResponse.SC_OK, path);
    }

    /**
     * Determine if the user has moderate permissions on the content of the request. Throw an exception if not.
     * @param request
     * @param session
     * @throws OperationException
     */
    protected void verifyUserHasModeratePermissions(final SlingHttpServletRequest request, final Session session)
        throws OperationException {
        final Resource resource = getSessionResource(request, session);
        if (!socialUtils.hasModeratePermissions(resource)) {
            throw new OperationException("Requesting user does not have permissions for this action.  ",
                HttpServletResponse.SC_UNAUTHORIZED);
        }
    }

    /**
     * Eliminates bulk items which the current session can't moderate.
     * @param bulkItemPaths array of paths of UGC resources
     * @param session current user session
     * @return a list of resources which the user can moderate
     * @throws LoginException occurs when session is invalid and thus can't retrieve the resource resolver
     */
    protected List<Resource> verifyUserBulkModerationPermissions(final String[] bulkItemPaths,
        final ResourceResolver resourceResolver) throws LoginException {
        final List<Resource> allowedResources = new ArrayList<Resource>();
        for (final String itemPath : bulkItemPaths) {
            final Resource resource = resourceResolver.resolve(itemPath);
            if (socialUtils.hasModeratePermissions(resource)) {
                allowedResources.add(resource);
            } else {
                log.warn("User '{}' doesn't have moderation permission for resource at '{}'",
                    resourceResolver.getUserID(), itemPath);
            }
        }
        return allowedResources;
    }

    /**
     * Determine if the user is a valid user. Throw an exception if not.
     * @param session The session the user is currently a part of.
     * @throws OperationException
     */
    protected void verifyUserIsValid(final Session session) throws OperationException {

        final String userId = session.getUserID();
        if (userId == null || userId.equalsIgnoreCase(CollabUser.ANONYMOUS)) {
            throw new OperationException("Requesting user does not have permissions for this action.  ",
                HttpServletResponse.SC_UNAUTHORIZED);
        }
    }

    /**
     * for unit testing..
     */
    public void setResourceResolverFactory(final ResourceResolverFactory aRRFactory) {
        resourceResolverFactory = aRRFactory;
    }

    /**
     * for unit testing..
     */
    public void setRepository(final SlingRepository aRepository) {
        repository = aRepository;
    }

    /**
     * for unit testing..
     */
    public void setModerationOperations(final ModerationOperations modOps) {
        moderationOps = modOps;
    }

    /**
     * for unit testing..
     */
    public void setSocialUtils(final SocialUtils aSocialUtils) {
        socialUtils = aSocialUtils;
    }

    /**
     * for unit testing..
     */
    public void setComponentFactoryManager(final SocialComponentFactoryManager aSocialComponentFactoryManager) {
        componentFactoryManager = aSocialComponentFactoryManager;
    }
}
