/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.commons.moderation.api;

import java.util.List;

import javax.jcr.Session;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;

import com.adobe.cq.social.commons.CommentException;

/**
 * @author mnew Interface for classes that perform moderation operations.
 */
public interface ModerationOperations {

    public static final String DISPLAY_FLAG_REASON_BOX = "social:displayFlagReasonBox";

    /**
     * Process a comment deny request sent by a client.
     * @param resource the resource to deny
     * @param session the session used to deny
     * @throws CommentException when the session cannot save the changes in the comment system
     */
    void deny(Resource resource, Session session) throws CommentException;

    /**
     * Process a bulk comment deny request sent by a client.
     * @param resourceList an array of resources to deny
     * @param session the user session requesting the bulk deny
     * @param resourceResolver the resource resolver
     * @throws CommentException for any exception, including when the session cannot save the change
     */
    void deny(List<Resource> resourceList, Session session, ResourceResolver resourceResolver)
        throws CommentException;

    /**
     * Process a comment flag request sent by a client.
     * @param resource the resource to flag
     * @param flagReason the reason the UGC is being flagged for moderation
     * @param doFlag flag if true, remove a flag if false
     * @param userId the id of the user flagging the comment
     * @throws CommentException when the session cannot save the changes in the comment system
     */
    void flag(Resource resource, String flagReason, boolean doFlag, String userId) throws CommentException;

    /**
     * Process an allow request sent by a client. Sets the UGC status back to okay.
     * @param resource the resource to allow
     * @param session the session used to allow
     * @param userId id of the originating user
     * @throws CommentException when the session cannot save the changes in the comment system
     */
    void allow(Resource resource, Session session, String userId) throws CommentException;

    /**
     * Process a bulk comment allow request sent by a client.
     * @param resourceList an array of resources to allow
     * @param session the user session requesting the bulk allow
     * @param resourceResolver the resource resolver
     * @throws CommentException for any exception, including when the session cannot save the change
     */
    void allow(List<Resource> resourceList, Session session, ResourceResolver resourceResolver)
        throws CommentException;

    /**
     * Process a close/open request sent by a client.
     * @param resource the resource to close
     * @param session the session used to close
     * @param close true if the request is to close content, false if it is to open.
     * @throws CommentException when the session cannot save the changes in the comment system
     */
    void close(Resource sessionResource, Session session, boolean close);

    /**
     * Process a bulk close/open request sent by a client.
     * @param resourceList an array of resources to close
     * @param session the session used to close
     * @param close true if the request is to close content, false if it is to open.
     * @throws CommentException when the session cannot save the changes in the comment system
     */
    void close(List<Resource> resourceList, Session session, ResourceResolver resolver, boolean close);

    /**
     * Process a delete request sent by a client. Sets the UGC status back to okay.
     * @param resource the resource to delete
     * @param session the session used to delete
     * @param resourceResolver the resource resolver
     * @throws CommentException when the session cannot save the changes in the comment system
     */
    void delete(Resource resource, Session session, ResourceResolver resourceResolver) throws CommentException;

    /**
     * Process a bulk comment delete request sent by a client.
     * @param resourceList an array of resources to delete
     * @param session the user session requesting the bulk delete
     * @param resourceResolver the resource resolver
     * @throws CommentException for any exception, including when the session cannot save the change
     */
    void delete(List<Resource> resourceList, Session session, ResourceResolver resourceResolver)
        throws CommentException;
}
