/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.cq.social.commons.tagging;

import java.util.Locale;
import java.util.Map;

import org.apache.sling.api.resource.Resource;

import com.adobe.cq.social.commons.tagging.impl.SocialTagImpl;
import com.day.cq.tagging.InvalidTagFormatException;
import com.day.cq.tagging.Tag;
import com.day.cq.tagging.TagManager;

/**
 * <code>SocialTagManager</code> is an extension to {@link TagManager} interface for Adobe Social.
 */
public interface SocialTagManager extends TagManager {

    // --------------------------------------------------------------< SoCo specific methods >
    /**
     * Verifies the creator of tag was indeed a user, not an admin nor moderator.
     * @param tagID can be the name/ID <code>mytag</code> or <code>namespace:group/tag</code> or the absolute path to
     *            resource.
     * @return true if a user created the tag, otherwise false
     */
    boolean isUserGeneratedTag(String tagID);

    /**
     * Returns true if the given tag was marked for moderation.
     * @param tagID can be the name/ID <code>mytag</code> or <code>namespace:group/tag</code> or the absolute path to
     *            resource.
     * @return true if tag was flagged for moderation, otherwise false or if tag was not user generated
     */
    boolean isFlaggedForModeration(String tagID);

    /**
     * Flags the given tag for moderation.
     * @param tag a user generated tag
     */
    void flagForModeration(Tag tag);

    /**
     * Removes the moderation flag of a tag.
     * @param tag a user generated tag that was flagged
     */
    void removeModerationFlag(Tag tag);

    /**
     * Verifies if the user, of the current session, can create a tag in the user generated space. This is not the
     * same as {@link TagManager}'s canCreateTag, which is not for user generated tags.
     * @param tagID can be the name/ID <code>mytag</code> or <code>namespace:group/tag</code> or the absolute path to
     *            resource.
     * @return true if the user has the needed permissions, false otherwise
     * @throws InvalidTagFormatException if tagID doen't adhere to formating rules
     */
    boolean canCreateUserGeneratedTag(String tagID) throws InvalidTagFormatException;

    /**
     * Creates a tag or namespace in the use generated space.
     * @param tagID Can be the absolute path or tag name/ID such as <code>myTag</code> or
     *            <code>namespace:group/myTag</code>. Namespaces can also be defined as absolute path or by the
     *            name/ID syntax as such <code>myNameSpace:</code>
     * @return the newly created tag
     * @throws InvalidTagFormatException if tagID doen't adhere to formating rules
     */
    Tag createUserGeneratedTag(String tagID) throws InvalidTagFormatException;

    /**
     * Retrieves all available tag namespaces, in the user generated space, as array.
     * @return all user generated tag namespaces
     */
    Tag[] getUserGeneratedNamespaces();

    /**
     * Add on or multiple tags to a resource.
     * @param resource an existing resource that may already have tags
     * @param tags an array of tags
     */
    void addTagsTo(Resource resource, Tag[] tags);

    /**
     * Removes one or multiple tags from a resource.
     * @param resource an resource an existing resource
     * @param tags an array of tags
     */
    void removeTagsFrom(Resource resource, Tag[] tags);

    /**
     * Removes all tags from a resource.
     * @param resource the tags to remove all its tags
     */
    void removeAllTagsFrom(Resource resource);

    /**
     * Returns the tag based on the resource at the given exact repository path. Does not handle if the tag has been
     * moved.
     * @param absPath absolute path to tag resource, such as <code>/etc/tags/namespace/mytag</code>
     * @return a {@link SocialTagImpl} instance for the tag resources stored at the given path, <code>null</code> if
     *         not found
     */
    Tag resolveRawTagByPath(String absPath);

    /**
     * Resolves search words to tags.
     * @param searchWords an array of search terms which a user entered
     * @param isUserGenerated whether to resolve in the user generated path, if <code>true</code>, otherwise in the
     *            base CQ path.
     * @return An array of all matching tags from all namespaces. An empty array may be returned if no matching tags
     *         were found.
     */
    Tag[] searchWordsToTags(String[] searchWords, boolean isUserGenerated);

    /**
     * Resolves search words to tags, while taking in consideration of the locale.
     * @param searchWords an array of search terms which a user entered
     * @param isUserGenerated whether to resolve in the user generated path, if <code>true</code>, otherwise in the
     *            base CQ path.
     * @param locale the local to use while resolving the tags.
     * @return An array of all matching tags from all namespaces. An empty array may be returned if no matching tags
     *         were found.
     */
    Tag[] searchWordsToTags(String[] searchWords, boolean isUserGenerated, Locale locale);

    /**
     * Get the tag facet for the specified target
     * @param resource
     * @return a map whose keys are the tags, and the values are the count.
     */
    Map<Tag, Integer> getTagFacetForSubtree(final Resource resource);
}
