/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.enablement.services.api;

import java.util.Map;

import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;

import com.adobe.cq.social.community.api.CommunityContext;
import com.adobe.cq.social.scf.OperationException;

/**
 * Provides methods to call when a Community Site needs to be enhanced with Enablement-specific functionality (i.e.
 * creation of additional groups and folders). Methods are called typically after the Site has been created or
 * updated.
 */
public interface EnablementSiteService {

    /**
     * Create default Enablement-specific user groups for a Community Site.
     * @param site the site's root resource
     * @param context community Context
     * @param session user session
     * @param properties request parameters
     * @throws OperationException - thrown if unable to create site user groups
     * @throws RepositoryException - thrown if unable to create site user groups
     */
    void createSiteUserGroups(final Resource site, final CommunityContext context, final Session session,
        final Map<String, Object> properties) throws RepositoryException, OperationException;

    /**
     * Apply Enablement-specific ACLs on Community Site resources.
     * @param resource the site's root resource
     * @param context community Context
     * @param session user session
     * @throws OperationException - thrown if unable to apply ACLs
     * @throws RepositoryException - thrown if unable to apply ACLs
     */
    void applyACLs(final Resource resource, final CommunityContext context, final Session session)
        throws OperationException, RepositoryException;

    /**
     * Apply Enablement-specific ACLs on Community Site resources.
     * @param site - resource the site's root resource
     * @param context community Context
     * @param aclSession session with ACL management permissions on site content.
     * @param userManagerSession session with permissions to lookup site users and groups.
     * @throws OperationException - thrown if unable to apply ACLs
     * @throws RepositoryException - thrown if unable to apply ACLs
     */
    void applyACLs(Resource site, CommunityContext context, Session aclSession, Session userManagerSession)
        throws RepositoryException, OperationException;

    /**
     * Create default Enablement-specific folders for a Community Site.
     * @param site the site's root resource
     * @param context community Context
     * @param session user session
     * @param properties request parameters
     * @throws OperationException - thrown if unable to create site folders
     * @throws RepositoryException - thrown if unable to create site folders.
     */
    @Deprecated
    void createSiteFolders(final Resource site, final CommunityContext context, final Session session,
        final Map<String, Object> properties) throws RepositoryException, OperationException;

    /**
     * Create default Enablement-specific folders for a Community Site.
     * @param site the site's root resource
     * @param context community Context
     * @param session user session, a session with read/write and replicate access to /content.
     * @param userManagerResolver a resolver for a user with access to manage users.
     * @param properties request parameters
     * @throws OperationException - thrown if unable to create site folders
     * @throws RepositoryException - thrown if unable to create site folders.
     */
    void createSiteFolders(final Resource site, final CommunityContext context, final Session session,
        final ResourceResolver userManagerResolver, final Map<String, Object> properties) throws RepositoryException,
        OperationException;

    /**
     * Create Enablement-specific site UGC nodes (i.e. /content/usergenerated/content/siteId/resources/langId) and set
     * the corresponding ACLs
     * @param resource site's root resource
     * @param context community context
     * @param session session with permissions to post to the site, read/write UGC nodes and change/add ACLs on site
     *            UGC nodes.
     * @throws OperationException - thrown if unable to create site UGC node(s)
     * @throws RepositoryException - thrown if unable to create site UGC node(s)
     */
    void createSiteUGCPath(final Resource resource, final CommunityContext context, final Session session)
        throws OperationException, RepositoryException;

    /**
     * Create Enablement-specific site UGC nodes (i.e. /content/usergenerated/content/siteId/resources/langId) and set
     * the corresponding ACLs
     * @param site - site's root resource must be from a resolver with permissions to post to the site and read/write
     *            site UGC nodes
     * @param context community context
     * @param aclSession session with permissions to change/add ACLs to the site UGC nodes
     * @param userManagerSession session with permissions to lookup site groups and members
     * @throws OperationException - thrown if unable to create site UGC node(s)
     * @throws RepositoryException - thrown if unable to create site UGC node(s)
     */
    void createSiteUGCPath(Resource site, CommunityContext context, Session aclSession, Session userManagerSession)
        throws OperationException, RepositoryException;

    /**
     * Update Enablement-specific user groups for a Community Site.
     * @param site site's root resource
     * @param context community Context
     * @param session user session
     * @param properties request parameters
     * @throws OperationException - thrown if unable to create site user groups
     * @throws RepositoryException - thrown if unable to create site user groups
     */
    void updateSiteUserGroups(final Resource site, final CommunityContext context, final Session session,
        final Map<String, Object> properties) throws RepositoryException, OperationException;

    /**
     * Checks whether the current instance is running in Publisher mode.
     * @return whether running in Publisher mode
     */
    boolean isPublishMode();

    /**
     * Gets the Enablement Assets path for the given Site.
     * @param context {@code CommunityContext}
     * @return Enablement Assets path
     */
    String getSiteAssetsPath(CommunityContext context);

    /**
     * Gets the Enablement Resources path for the given context.
     * @param context {@code CommunityContext}
     * @return Enablement Resources path
     */
    String getSiteResourcesPath(CommunityContext context);

    /**
     * Gets the Enablement Resource-groups path for the given context.
     * @param context {@code CommunityContext}
     * @return Enablement Resource-groups path
     */
    String getSiteResourceGroupsPath(CommunityContext context);
}
