/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2015 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 *
 *************************************************************************/
package com.adobe.cq.social.enablement.services.api;

import javax.jcr.RepositoryException;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.commons.json.JSONException;
import org.apache.sling.commons.json.JSONObject;

import com.adobe.cq.social.enablement.model.EnablementResource;
import com.adobe.cq.social.enablement.utils.ResourceUtils;
import com.adobe.cq.social.tally.client.api.TallyException;

public class ResourceDetailsImpl {
    public static final String RESOURCE_INFO = "info";
    public static final String ASSET_ARRAY = "assets";
    public static final String RATINGS_OBJECT = "ratings";

    /**
     * Get the resource's detail at the pointed to by the resource.
     * @param resource {@link Resource} for which {@link com.adobe.cq.social.enablement.model.EnablementResource} 's
     *            details are required
     * @return {@link JSONObject} with following properties:
     *         <ul>
     *         <li>{@link #RESOURCE_INFO}: value of 1st level properties of resource node in JCR</li>
     *         <li>{@link #ASSET_ARRAY}: value as an array of assets participating in this resource. This array would
     *         consist of objects with properties:
     *         <ul>
     *         <li>ASSET_NAME</li>
     *         <li>ASSET_END_POINT</li>
     *         <li>ASSET_COVER_IMAGE_URL</li>
     *         <li>ASSET_MIME_TYPE</li>
     *         </ul>
     *         </li>
     *         <li>{@link #RATINGS_OBJECT}: this object would have following rating related properties:
     *         <ul>
     *         <li>RATINGS_USER: rating by the current user as a number</li>
     *         <li>RATINGS_USER_STRING: string message representing rating by the current user</li>
     *         <li>RATINGS_AVG: average rating as a number</li>
     *         <li>RATINGS_COUNT: count of ratings</li>
     *         </ul>
     *         </li>
     *         <li>USER_INTERATION_WATCHED: true/false (missing property implies false)</li>
     *         <li>USER_INTERATION_PINNED: true/false (missing property implies false)</li>
     *         </ul>
     *         <p>
     *         &nbsp;
     *         </p>
     *         Sample Response:
     *         <p>
     *         &nbsp;
     *         </p>
     *
     *         <pre>
     * {
     *     //1st level resource information saved in a resource node in JCR(essentially &lt;resource-end-point&gt;.json)
     *     "info": {
     *         "jcr:primaryType": "nt:unstructured",
     *         "jcr:mixinTypes": ["rep:AccessControllable"],
     *         "allow-social-switch": false,
     *         "resource-experts": [],
     *         "description": "Tutorial on AEM",
     *         "cover-image-path": "./cover-image/file/image",
     *         "resource-managers": [],
     *         "allow-ratings": false,
     *         "date-modified": "Tue Feb 11 2014 19:19:05 GMT+0530",
     *         "is-mandatory": false,
     *         "publish": true,
     *         "id": "101",
     *         "url": "",
     *         "name": "AEM Training2",
     *         "tags": ["adobe:social"],
     *         "category": ["adobe:aem"],
     *         "resource-contact": [],
     *         "date-created": "Tue Feb 11 2014 19:19:05 GMT+0530",
     *         "moderate-comments": true,
     *         "catalog-visibility": "open",
     *         "created-by": "nikeadmin",
     *         "allow-comments": false
     *     },
     *     //asset list for this resource with some meta-information about asset.
     *     //The meta info about asset is basically with guidance that what information is relevant to be
     *     //shown while detailing asset(suggestion needed--a few guesses by me: duration, pages[if applicable]
     *     //more specific info about asset(like video progress) need to be pulled from asset end-point
     *     "assets": [ //0..*
     *         {
     *             "name": "KB2.mp4",
     *             "end-point": "/content/enablement/data/nike/resources/nikeadmin/AEMTraining/assets/asset1/jcr:content",
     *             "cover-image-path": "/content/enablement/data/nike/resources/nikeadmin/AEMTraining/assets/asset1/cover-img/image",
     *             "type": "video/mp4",
     *         },
     *         {
     *             "name": "IMG_3418.jpg",
     *             "end-point": "/content/enablement/data/nike/resources/nikeadmin/AEMTraining/assets/asset2/jcr:content",
     *             "cover-image-path": "/content/enablement/data/nike/resources/nikeadmin/AEMTraining/assets/asset2/cover-img/image",
     *             "type": "image/jpeg",
     *         }
     *     ],
     *     "ratings": {
     *             "count": 1,
     *             "average": 3,
     *             "user-rating": 3, //(optional)
     *             "user-rating-string": "3.0" //(optional)
     *     }
     * }
     * </pre>
     * @throws RepositoryException -- throws exception when unable to find the resource
     * @throws TallyException -- generic exception for all failures in the Tally API.
     * @throws JSONException -- throws when there are issue with the JSON
     */
    public JSONObject getResourceDetails(Resource resource) throws RepositoryException, TallyException, JSONException {
        EnablementResource enablementResource = resource.adaptTo(EnablementResource.class);
        JSONObject resourceDetails = new JSONObject();

        // TODO remove the tags and categgories object from resourceInfoObj to avoid duplication.
        JSONObject resourceInfoObj = resource.adaptTo(JSONObject.class);

        resourceDetails.put(RESOURCE_INFO, resourceInfoObj);
        resourceDetails.put(ASSET_ARRAY, ResourceUtils.getResourceAssets(enablementResource));
        resourceDetails.put(RATINGS_OBJECT, ResourceUtils.getResourceRating(enablementResource));
        return resourceDetails;
    }
}
