/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.filelibrary.client.api;

import javax.jcr.RepositoryException;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceUtil;
import org.apache.sling.api.resource.ValueMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.cq.social.commons.CommentSystem;
import com.adobe.cq.social.commons.comments.api.AbstractComment;
import com.adobe.cq.social.commons.comments.listing.CommentSocialComponentListProviderManager;
import com.adobe.cq.social.scf.ClientUtilities;
import com.adobe.cq.social.scf.CollectionPagination;
import com.adobe.cq.social.scf.QueryRequestInfo;
import com.adobe.cq.social.scf.SocialComponent;
import com.adobe.cq.social.scf.core.DefaultResourceID;
import com.adobe.cq.social.scf.core.ResourceID;

public abstract class AbstractAsset<T extends FileLibraryConfiguration> extends AbstractComment<T> implements
    Asset<T> {

    private static final String TOPIC_HTML_SUFFIX = ".asset.html";
    private final String name;
    private final Resource parent;
    protected ValueMap properties;
    protected final boolean isFolder;
    private CommentSocialComponentListProviderManager listProviderManager;
    private static final Logger LOG = LoggerFactory.getLogger(AbstractAsset.class);

    public AbstractAsset(final Resource resource, final ClientUtilities clientUtils,
        final CommentSocialComponentListProviderManager listProviderManager) throws RepositoryException {
        this(resource, clientUtils, QueryRequestInfo.DEFAULT_QUERY_INFO_FACTORY.create(), listProviderManager);
    }

    /**
     * Constructor of an asset.
     * @param resource the specified resource
     * @param clientUtils the client utilities instance
     * @param pagination the list pagination configuration of this comment
     * @param commentFactory comment factory used to create a legacy comment instance to a Post
     * @throws RepositoryException if an error occurs
     */
    public AbstractAsset(final Resource resource, final ClientUtilities clientUtils,
        final QueryRequestInfo queryInfo, final CommentSocialComponentListProviderManager listProviderManager)
        throws RepositoryException {
        super(resource, clientUtils, queryInfo, listProviderManager);
        properties = ResourceUtil.getValueMap(resource);
        name = properties.get(Asset.PN_NAME, String.class);
        parent = FileLibraryUtils.getParent(resource, false);
        isFolder = resource.getResourceResolver().isResourceType(resource, FileLibrary.RESOURCE_TYPE_FOLDER);
        this.listProviderManager = listProviderManager;
    }

    /**
     * @inheritDoc
     */
    @Override
    public String getName() {
        return name;
    }

    /**
     * @inheritDoc
     */
    @Override
    public boolean isFolder() {
        return isFolder;
    }

    /**
     * @inheritDoc
     */
    @Override
    public String getFileLibraryId() {
        return getParentId();
    }

    @Override
    public boolean isFileLibraryClosed() {
        return false;
    }

    @Override
    public String getFriendlyUrl() {
        final String pagePath = clientUtils.getSocialUtils().getContainingPage(parent).getPath();
        ResourceID urlid = this.id;
        if (!isFolder()) {
            urlid = new DefaultResourceID(getResource());
        }
        // TODO:
        return clientUtils.externalLink(pagePath, false) + TOPIC_HTML_SUFFIX + urlid + ".html";
    }

    @Override
    public void setPagination(final CollectionPagination pagination) {
        super.setPagination(pagination);
    }

    @Override
    public T createConfiguration(final Resource resource, final Resource cs) {
        return (T) new AbstractFileLibraryConfiguration(cs);
    }

    @Override
    public String getParentName() {
        final SocialComponent parent = this.getParentComponent();
        if (parent == null || !(parent instanceof Asset)) {
            return null;
        }
        final Asset parentAsset = (Asset) this.getParentComponent();
        return parentAsset.getName();
    }

    @Override
    public String getParentFriendlyUrl() {
        final SocialComponent parent = this.getParentComponent();
        if (parent == null) {
            return null;
        }
        return parent.getFriendlyUrl();
    }
}
