/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.forum.client.api;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.commons.lang3.StringUtils;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceUtil;
import org.apache.sling.api.resource.ValueMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.cq.social.commons.CommentSystem;
import com.adobe.cq.social.commons.comments.api.AbstractComment;
import com.adobe.cq.social.commons.comments.listing.CommentSocialComponentListProviderManager;
import com.adobe.cq.social.forum.client.impl.ForumUtils;
import com.adobe.cq.social.forum.client.impl.PostImpl;
import com.adobe.cq.social.scf.ClientUtilities;
import com.adobe.cq.social.scf.CollectionPagination;
import com.adobe.cq.social.scf.QueryRequestInfo;
import com.adobe.cq.social.scf.SocialComponent;
import com.adobe.cq.social.scf.User;
import com.adobe.cq.social.scf.core.DefaultResourceID;
import com.adobe.cq.social.scf.core.ResourceID;
import com.adobe.cq.social.ugcbase.SocialUtils;
import com.day.cq.wcm.api.Page;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonInclude.Include;

public abstract class AbstractPost<T extends ForumConfiguration> extends AbstractComment<T> implements Post<T> {

    private static final String CQ_META_NODE_NAME = "cq:meta";
    private static final String PROP_VIEWS = "views";
    private static final String SEARCH_SUFFIX_LONG = "_tl";
    private static final String TOPIC_HTML_SUFFIX = ".topic.html";
    private Resource latestPost;
    private final Resource parent;
    private int numReplies;
    private ValueMap properties;
    private boolean isTopic;
    private final int numParticipants = -1;
    private User lastModifiedUser;
    private CommentSocialComponentListProviderManager listProviderManager;
    private boolean isAnalyticsEnabled;
    private static final Logger LOG = LoggerFactory.getLogger(PostImpl.class);

    public AbstractPost(final Resource resource, final ClientUtilities clientUtils,
        final CommentSocialComponentListProviderManager listProviderManager) throws RepositoryException {
        this(resource, clientUtils, QueryRequestInfo.DEFAULT_QUERY_INFO_FACTORY.create(), listProviderManager);
    }

    /**
     * Constructor of a comment.
     * @param resource of the component
     * @param clientUtils the client utilities instance
     * @param queryInfo The query request Info
     * @param listProviderManager the list provider manager
     * @throws RepositoryException if an error occurs
     */
    public AbstractPost(final Resource resource, final ClientUtilities clientUtils, final QueryRequestInfo queryInfo,
        final CommentSocialComponentListProviderManager listProviderManager) throws RepositoryException {
        super(resource, clientUtils, queryInfo, listProviderManager);
        properties = ResourceUtil.getValueMap(resource);
        isTopic = isTopLevel();
        parent = ForumUtils.getParent(resource, isTopic, false);
        this.listProviderManager = listProviderManager;
        isAnalyticsEnabled = clientUtils.isAnalyticsServiceConfigured(resource);
    }

    public AbstractPost(final Resource resource, final ClientUtilities clientUtils, final QueryRequestInfo queryInfo,
        final Resource latestPost, final int numReplies,
        final CommentSocialComponentListProviderManager listProviderManager) throws RepositoryException {
        this(resource, clientUtils, queryInfo, listProviderManager);
        this.latestPost = latestPost;
        this.numReplies = numReplies;
        this.listProviderManager = listProviderManager;
    }

    @Override
    public String getSubject() {
        final String subject = properties.get(com.adobe.cq.social.forum.api.Post.PN_SUBJECT, String.class);
        if (isTopic) {
            return subject;
        } else {
            if (StringUtils.isEmpty(subject)) {
                return properties.get(com.adobe.cq.social.forum.api.Post.PN_SUBJECT, String.class);
            } else {
                return subject;
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    @JsonInclude(value = Include.NON_NULL)
    public User getLastModifiedUser() {
        if (lastModifiedUser == null) {
            // final String userId = properties.get(LAST_SELF_MOD_BY, super.getAuthor().getId());
            String userId = null;
            if (latestPost != null) {
                Post lpost;
                try {
                    lpost = new PostImpl(latestPost, clientUtils, getQueryRequestInfo(), listProviderManager);
                    userId = lpost.getAuthor().getAuthorizableId();
                } catch (final RepositoryException e) {
                    LOG.error("error getting latest post.", e);
                }
            }
            if (userId == null) {
                userId = super.getAuthor().getAuthorizableId();
            }
            lastModifiedUser = clientUtils.getUser(userId, resource.getResourceResolver());
        }
        return lastModifiedUser;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isPinned() {
        return properties.get(com.adobe.cq.social.forum.api.Post.PN_PINNED, false);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isClosed() {
        return (isForumClosed() || isTopicClosed());
    }

    /**
     * @return true if the topic is closed (this is based off AbstractComment).
     */
    @Override
    public boolean isTopicClosed() {
        return super.isClosed();
    }

    /**
     * @return true if the entire forum is closed.
     */
    @Override
    public boolean isForumClosed() {
        final SocialComponent sc = getSourceComponent();
        if (sc == null) {
            return false;
        }
        return ((com.adobe.cq.social.forum.client.api.Forum) sc).isClosed();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isTopic() {
        return isTopic;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getForumId() {
        final SocialComponent sc = getSourceComponent();
        if (sc == null) {
            return null;
        }
        return sc.getId().getResourceIdentifier();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    @JsonInclude(value = Include.NON_NULL)
    public String getLatestPost() {
        return (latestPost == null) ? null : latestPost.getPath();
    }

    // /**
    // * Get total size of the collection.
    // * @return the size of the collection.
    // */
    // public int getTotalSize() {
    // return numReplies;
    // }

    @Override
    public void setLatestPost(final Resource post) {
        this.latestPost = post;
    }

    @Override
    public void setNumReplies(final int replies) {
        this.numReplies = replies;
    }

    @Override
    public String getFriendlyUrl() {
        final String pagePath;
        final Page page = (parent == null) ? null : clientUtils.getSocialUtils().getContainingPage(parent);
        if (page != null) {
            pagePath = page.getPath();
        } else {
            if (parent == null) {
                LOG.error("Can not find containing page because parent is null.");
            } else {
                LOG.error("Can not find containing page for {}.", parent.getPath());
            }
            pagePath = null;
        }

        ResourceID urlid = this.id;
        if (!isTopic) {
            final Resource topicResource = ForumUtils.getParent(getResource(), isTopic, true);
            urlid = new DefaultResourceID(topicResource);
        }
        return clientUtils.externalLink(
            pagePath + TOPIC_HTML_SUFFIX + "/" + StringUtils.substringAfterLast(urlid.getResourceIdentifier(), "/"),
            false) + ".html";
    }

    @Override
    public void setPagination(final CollectionPagination pagination) {
        super.setPagination(pagination);
    }

    @Override
    protected T createConfiguration(final Resource resource, final Resource commentSystem) {
        return (T) new BaseForumConfiguration(commentSystem);
    }

    @Override
    public Map<String, Integer> getViewCounts() {
        Map<String, Integer> viewCounts = new HashMap<String, Integer>();
        if (isAnalyticsEnabled) {
            Resource analyticsMetaData = resource.getChild(CQ_META_NODE_NAME);
            if (analyticsMetaData != null) {
                Iterator<Resource> iter = analyticsMetaData.getChildren().iterator();
                while (iter.hasNext()) {
                    Resource reportConfig = iter.next();
                    ValueMap vm = reportConfig.getValueMap();
                    viewCounts.put(reportConfig.getName(),
                        vm.get(PROP_VIEWS + "_" + reportConfig.getName() + SEARCH_SUFFIX_LONG, 0));
                }
            }
        }
        return viewCounts;
    }

    @Override
    protected boolean canUserReply(final boolean userIsLoggedIn, final CommentSystem cs, final Session session) {
        if (!userIsLoggedIn) {
            return false;
        }
        boolean mayPost = false;
        final SocialUtils socialUtils = clientUtils.getSocialUtils();
        if (socialUtils != null) {
            final String aclPath = socialUtils.resourceToACLPath(cs.getResource());
            mayPost =
                clientUtils.getSocialUtils().canAddNode(session, aclPath == null ? CommentSystem.PATH_UGC : aclPath);

        } else {
            return false;
        }
        if (isTopLevel()) {
            return !isClosed() && mayPost;
        } else {
            return cs.allowsReplies() && !isClosed() && mayPost;
        }
    }
}
