/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.forum.client.api;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Dictionary;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.osgi.service.event.Event;

import com.adobe.cq.social.commons.events.CommentEvent.CommentActions;
import com.adobe.cq.social.forum.client.api.ForumEvent.Type;
import com.adobe.cq.social.scf.ClientUtilities;
import com.adobe.cq.social.scf.SocialComponent;
import com.adobe.cq.social.scf.core.SocialEvent;
import com.adobe.granite.activitystreams.ObjectTypes;
import com.adobe.granite.activitystreams.Verbs;

/**
 * Event fired for changes occurring within a forum.
 * @since CQ 5.6.0
 */
public final class ForumEvent extends SocialEvent<Type> implements Serializable {

    /** A serialVersionUID for this class. */
    private static final long serialVersionUID = 2L;

    /**
     * Property for event type.
     */
    private static final String PROP_EVENT_TYPE = "EventType";

    /**
     * Property for event path.
     */
    private static final String PROP_EVENT_PATH = "EventPath";

    /**
     * Property for forum path.
     */
    private static final String PROP_FORUM_PATH = "ForumPath";

    public static final String FORUM_TOPIC = "forum";

    /**
     * The topic name for event topic.
     */
    public static final String EVENT_TOPIC = SocialEvent.SOCIAL_EVENT_TOPIC_PREFIX + FORUM_TOPIC;

    /**
     * We must handle collab events to support notifications in collab forums. The old code that collabs depended on
     * for notifications in 5.5 no longer exists. ForumEventBuilder has been moved to social-forum so we need to
     * handle collab notifications in social.
     * @deprecated - this has been deprecated, we no longer need to be triggering collab events from social bundles.
     */
    @Deprecated
    public static final String COLLAB_EVENT_TOPIC = "com/day/cq/collab/forum";

    protected static final String PROP_TOPIC_PATH = "TopicPath";

    /**
     * The event types supported for the forum and its topics/posts.
     */
    public static enum Type implements com.adobe.cq.social.scf.core.SocialEvent.SocialActions {
        /**
         * The event fired upon a topic being added.
         */
        TopicAdded,

        /**
         * The event fired upon a post being added.
         */
        PostAdded,

        /**
         * The event fired upon approval of a moderated post.
         * @deprecated This event type is never used (and never will be).
         */
        @Deprecated
        PostApproved,

        TopicEdited,

        PostEdited,

        TopicDeleted,

        PostDeleted;

        private static final Set<String> FORUM_ACTIONS = new HashSet<String>();
        static {
            for (Type action : Type.values()) {
                FORUM_ACTIONS.add(FORUM_TOPIC + "." + action.getVerb());
            }
        }

        @Override
        public String getVerb() {
            switch (this) {
                case TopicAdded:
                    return Verbs.POST;
                case PostAdded:
                    return Verbs.ADD;
                case PostApproved:  // deprecated
                    return Verbs.APPROVE;
                case TopicEdited:
                case PostEdited:
                    return Verbs.UPDATE;
                case TopicDeleted:
                case PostDeleted:
                    return Verbs.DELETE;
                default:
                    throw new IllegalArgumentException();
            }
        }
    }

    private static Map<String, Object> getPropertiesMap(final Dictionary<String, Object> properties) {
        final Map<String, Object> additionalProps = new HashMap<String, Object>(properties.size());
        final Enumeration<String> keys = properties.keys();
        while (keys.hasMoreElements()) {
            final String key = keys.nextElement();
            if (!(PROP_EVENT_PATH.equals(key) || PROP_EVENT_TYPE.equals(key) || com.adobe.cq.social.scf.core.SocialEvent.USER_ID
                .equals(key))) {
                additionalProps.put(key, properties.get(key));
            }
        }
        return additionalProps;
    }

    private ForumEvent(final Event event) {
        super(event);
    }

    public ForumEvent(final Post post, final String user, final Type action) {
        super(FORUM_TOPIC, post.getResource().getPath(), user, action, post.isTopic() ? buildTopicProperties(post,
            action) : buildPostProperties(post, action));
    }

    private static Map<String, Object> buildTopicProperties(final Post post, final Type action) {
        if (action.getVerb().equals(Verbs.DELETE)) {
            return new HashMap<String, Object>(2) {
                {
                    this.put(OBJECT,
                        new BaseEventObject("a topic", post.getResource().getPath(), ObjectTypes.COMMENT));
                    this.put(TARGET, new BaseEventObject(getDisplayName(post.getParentComponent()),
                        post.getForumId(), ObjectTypes.COLLECTION));
                };
            };
        } else {
            return new HashMap<String, Object>(2) {
                {
                    this.put(OBJECT, new BaseEventObject(getDisplayName(post), post.getResource().getPath(),
                        ObjectTypes.COMMENT));
                    this.put(TARGET, new BaseEventObject(getDisplayName(post.getParentComponent()),
                        post.getForumId(), ObjectTypes.COLLECTION));
                };
            };
        }
    }

    private static Map<String, Object> buildPostProperties(final Post post, final Type action) {
        return new HashMap<String, Object>(2) {
            {
                this.put(OBJECT, new BaseEventObject((action.getVerb().equals(Verbs.DELETE)) ? "a post"
                    : getDisplayName(post), post.getResource().getPath(), ObjectTypes.COMMENT));
                SocialComponent parent = post.getParentComponent();
                this.put(TARGET, new BaseEventObject(getDisplayName((Post) parent), parent.getResource().getPath(),
                    ObjectTypes.COLLECTION));
            };
        };
    }

    private static String getDisplayName(final SocialComponent post) {
        if (post instanceof Post) {
            final String name = ((Post) post).getSubject();
            if (StringUtils.isNotEmpty(name)) {
                return name;
            } else {
                return ((Post) post).isTopic() ? "a topic" : "a post";
            }
        } else {
            if (post instanceof Forum) {
                return ((Forum) post).getTitle();
            } else {
                return null;
            }
        }
    }

    protected static String getUserId(final String userId) {
        if (StringUtils.isNotEmpty(userId) && userId.startsWith(ClientUtilities.USER_ROOTPATH)) {
            return userId.substring(ClientUtilities.USER_ROOTPATH.length());
        } else {
            return userId;
        }
    }
}
