/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.forum.client.api;

import java.util.Calendar;
import java.util.Map;

import org.apache.sling.api.resource.Resource;

import com.adobe.cq.social.commons.comments.api.Comment;
import com.adobe.cq.social.commons.comments.api.CommentCollectionConfiguration;
import com.adobe.cq.social.commons.comments.api.PageInfo;
import com.adobe.cq.social.scf.User;

/**
 * The <code>Post</code> represents a topic of a forum, or a post of a topics. The {@link #isTopic() isTopic} method
 * indicates whether an instance of a Post is a topic or not.
 * @param <T> the generic type
 */
public interface Post<T extends CommentCollectionConfiguration> extends Comment<T> {
    /**
     * JCR properties that maintains the id of the user that effected the last modification of this topic.
     */
    final String LAST_SELF_MOD_BY = "cq:lastSelfModifiedBy";

    /** The pn subject. */
    final String PN_SUBJECT = "jcr:title";

    /** The post postfix. */
    final String POST_POSTFIX = "_tosp";

    /** The property that is used in request when creating post on behalf of a different user */
    final String PN_COMPOSED_FOR = "composedFor";

    /** The property that stores the original author */
    final String PN_COMPOSED_BY = "composedBy";

    /**
     * This method returns the forum post's subject as entered by the user. In general only top-level posts (topics)
     * have a subject. If the post doesn't have a subject, the subject of top level parent (topic) is returned.
     * @return A <code>String</code> representing the post's subject or <code>null</code> if not defined.
     */
    String getSubject();

    /**
     * Returns the user that effected the last modification of this post.
     * @return A <code>User</code> last modified this post.
     */
    User getLastModifiedUser();

    /**
     * Indicates whether this post is pinned. A post is pinned if it is a topic (top level post) or its parent topic
     * is pinned.
     * @return <code>true</code> if this post is pinned.
     */
    boolean isPinned();

    /**
     * Indicates whether this forum or topic is closed for further addition of topics/posts.
     * @return <code>true</code> if this topic is closed.
     */
    @Override
    boolean isClosed();

    /**
     * Check if this instance is a Post or not.
     * @return <code>true</code> if this post is a topic.
     */
    boolean isTopic();

    /**
     * Get the forum id for this post.
     * @return the forum id.
     */
    String getForumId();

    /**
     * Returns the latest post, i.e. the post with the most recent modification date. If this post is not a topic or
     * doesn't have any posts, <code>null</code> is returned.
     * @return The latest post of this topic or <code>null</code> if not found.
     */
    String getLatestPost();

    /**
     * Gets the friendly url.
     * @return a search engine friendly url to render this post if this post is a topic. if the post is a topic, this
     *         is blank.
     */
    @Override
    String getFriendlyUrl();

    /**
     * Set latest post.
     * @param post the latest post.
     */
    void setLatestPost(final Resource post);

    /**
     * Set the number of replies.
     * @param replies the latest number of replies.
     */
    void setNumReplies(final int replies);

    /**
     * Gets information about the pages for this collection. This can be used by the page block helper to easily
     * render various pagination UIs.
     * @return information about the pagination system
     */
    @Override
    PageInfo getPageInfo();

    /**
     * Checks if is topic closed.
     * @return true if the topic is closed.
     */
    boolean isTopicClosed();

    /**
     * Checks if is forum closed.
     * @return true if the entire forum is closed.
     */
    boolean isForumClosed();

    /**
     * Gets the view counts for this post by time segment.
     * @return
     */
    Map<String, Integer> getViewCounts();
}
