/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.cq.social.forum.client.endpoints;

import java.util.Map;

import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang.StringUtils;
import org.apache.felix.scr.annotations.Component;
import org.apache.sling.api.SlingConstants;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceUtil;
import org.apache.sling.api.resource.ValueMap;

import com.adobe.cq.social.commons.Comment;
import com.adobe.cq.social.commons.CommentSystem;
import com.adobe.cq.social.commons.comments.endpoints.AbstractCommentOperationService;
import com.adobe.cq.social.forum.client.api.Forum;
import com.adobe.cq.social.forum.client.api.Post;
import com.adobe.cq.social.scf.Operation;
import com.adobe.cq.social.scf.OperationException;
import com.adobe.cq.social.scf.OperationExtension;
import com.adobe.cq.social.scf.SocialComponent;
import com.adobe.cq.social.scf.SocialComponentFactory;
import com.adobe.cq.social.scf.SocialComponentFactoryManager;
import com.adobe.cq.social.ugcbase.core.SocialResourceUtils;
import com.day.cq.wcm.api.designer.Designer;
import com.day.cq.wcm.api.designer.Style;

/**
 * Common Sling Operation for forum.
 * @param <T> is a {@link OperationExtension}
 * @param <U> is a {@link Operation}
 */
@Component(metatype = false, componentAbstract = true)
public abstract class AbstractForumOperationsService<T extends OperationExtension, U extends Operation, S extends Post>
    extends AbstractCommentOperationService<T, U, S> {

    @Override
    protected boolean mayReply(final Resource root, final CommentSystem cs) throws OperationException {
        boolean retVal = false;
        if (root != null) {
            final ResourceResolver resolver = root.getResourceResolver();
            retVal = resolver.isResourceType(root, Forum.RESOURCE_TYPE_TOPIC);
            if (!retVal && resolver.isResourceType(root, Forum.RESOURCE_TYPE_POST)) {
                retVal = cs.allowsReplies();
            }

        }
        return retVal;
    }

    @Override
    protected boolean mayPost(final SlingHttpServletRequest request, final CommentSystem cs, final String userId) {
        boolean retVal = super.mayPost(request, cs, userId);

        final boolean isClosed = cs.getProperty(com.adobe.cq.social.forum.api.Forum.PN_CLOSED, false);
        retVal = retVal && !isClosed;
        if (cs.getResource().getPath().equals(request.getResource().getPath())) {
            // if topic check for privilieged user
            retVal = retVal && isUserPrivileged(cs, getRequestSession(request), userId);
        }
        return retVal;
    }

    @Override
    protected void getDefaultProperties(final SlingHttpServletRequest request, final Map<String, Object> props,
        final Session session) throws RepositoryException, OperationException {
        super.getDefaultProperties(request, props, session);
        final String subject = request.getParameter(Forum.REQ_PN_SUBJECT);
        if (!StringUtils.isBlank(subject)) {
            props.put(Post.PN_SUBJECT, subject);
        }
    }

    @Override
    protected String getResourceType(final Resource root) {
        if (ResourceUtil.isA(root, Forum.RESOURCE_TYPE)) {
            return getForumTopicResourceType(root);
        } else {
            Comment comment = root.adaptTo(Comment.class);
            CommentSystem cs = comment.getCommentSystem();
            return getForumPostResourceType(cs.getResource());
        }
    }

    private String getForumPostResourceType(final Resource root) {
        String postResourceType = getForumPostResourceType();
        final Designer forumDesign = root.getResourceResolver().adaptTo(Designer.class);
        final ValueMap forumProperties = root.adaptTo(ValueMap.class);
        if (null != forumDesign) {
            final Style currentStyle = forumDesign.getStyle(root);
            // Only use the style if the resource types match, e.g. QnA shouldn't use design types from the forum
            if ((null != currentStyle)
                    && (StringUtils.equals(
                        (String) currentStyle.get(SlingConstants.NAMESPACE_PREFIX + ":"
                                + SlingConstants.PROPERTY_RESOURCE_TYPE),
                        forumProperties.get(SlingConstants.NAMESPACE_PREFIX + ":"
                                + SlingConstants.PROPERTY_RESOURCE_TYPE, String.class)))) {
                postResourceType =
                    (currentStyle != null) ? currentStyle.get(getPostDesignResourceType(), postResourceType)
                        : postResourceType;
            }
        }
        return postResourceType;
    }

    protected String getForumPostResourceType() {
        return Forum.RESOURCE_TYPE_POST;
    }

    private String getForumTopicResourceType(final Resource resource) {
        String topicResourceType = getForumTopicResourceType();
        final Designer forumDesign = resource.getResourceResolver().adaptTo(Designer.class);
        if (null != forumDesign) {
            final Style currentStyle = forumDesign.getStyle(resource);
            // Only use the style if the resource types match, e.g. QnA shouldn't use design types from the forum
            if ((null != currentStyle)
                    && (StringUtils.equals(
                        (String) currentStyle.get(SlingConstants.NAMESPACE_PREFIX + ":"
                                + SlingConstants.PROPERTY_RESOURCE_TYPE), resource.getResourceType()))) {
                topicResourceType =
                    (currentStyle != null) ? currentStyle.get(getTopicDesignResourceType(), topicResourceType)
                        : topicResourceType;
            }
        }
        return topicResourceType;
    }

    protected String getForumTopicResourceType() {
        return Forum.RESOURCE_TYPE_TOPIC;
    }

    @Override
    protected void validateParameters(final SlingHttpServletRequest request) throws OperationException {
        super.validateParameters(request);
        if (request.getResource().isResourceType(Forum.RESOURCE_TYPE)) {
            final String message = request.getParameter(Forum.REQ_PN_SUBJECT);
            if (message == null || "".equals(message)) {
                throw new OperationException("Topic Subject may not be empty", HttpServletResponse.SC_BAD_REQUEST);
            }
        }
    }

    @Override
    protected String getEntityUrl(final Resource resource) {
        // Get the component factory
        final SocialComponentFactory scf = getSocialComponentFactoryManager().getSocialComponentFactory(resource);
        if (scf != null) {
            if (SocialResourceUtils.isSocialResource(resource)) {
                final SocialComponent sc = scf.getSocialComponent(resource);
                if (sc != null) {
                    final Post<?> post = (Post<?>) sc;
                    if (post != null) {
                        return post.getFriendlyUrl();
                    }
                }
            }
        }
        return super.getEntityUrl(resource);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected abstract S getSocialComponentForResource(final Resource resource);

    /**
     * Get Forum design property name for topic.
     * @return property name for topic
     */
    public abstract String getTopicDesignResourceType();

    /**
     * Get Forum design property name for post.
     * @return property name for post
     */
    public abstract String getPostDesignResourceType();

    /**
     * Get {@link SocialComponentFactoryManager SocialComponentFactoryManager} for this operation service.
     * @return social component factory manager used in this forum.
     */
    public abstract SocialComponentFactoryManager getSocialComponentFactoryManager();
}
