/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2015 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.graph.client.api;

import java.util.ArrayList;
import java.util.List;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.apache.commons.lang.StringUtils;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ValueMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.cq.social.graph.Edge;
import com.adobe.cq.social.graph.SocialGraph;
import com.adobe.cq.social.graph.Vertex;
import com.adobe.cq.social.scf.ClientUtilities;
import com.adobe.cq.social.scf.CollectionPagination;
import com.adobe.cq.social.scf.QueryRequestInfo;
import com.adobe.cq.social.scf.SocialComponent;
import com.adobe.cq.social.scf.SocialComponentFactory;
import com.adobe.cq.social.scf.SocialComponentFactoryManager;
import com.adobe.cq.social.scf.User;
import com.adobe.cq.social.scf.core.BaseSocialComponent;
import com.adobe.cq.social.scf.core.CollectionSortedOrder;
import com.adobe.granite.security.user.UserProperties;
import com.adobe.granite.security.user.UserPropertiesManager;
import com.adobe.granite.socialgraph.Direction;

/**
 * Return list of INCOMING or OUTGOING relationship of the user session.
 * @author thuynh
 */
public class AbstractRelationshipCollection extends BaseSocialComponent implements RelationshipCollection {
    private final ResourceResolver resolver;
    private final SocialGraph graph;
    private final Direction direction;
    private final String relType;
    private String userId;
    private List<Object> socialComponents;
    private static final Logger LOG = LoggerFactory.getLogger(AbstractRelationshipCollection.class);
    private static String PROP_USERID = "userId";
    private static final String PROP_OUTGOING_DIRECTION = "outgoing";
    private static final String PROP_REL_TYPE = "relType";

    public AbstractRelationshipCollection(final Resource resource, final ClientUtilities clientUtils,
        final QueryRequestInfo requestInfo) {
        super(resource, clientUtils);
        resolver = resource.getResourceResolver();
        graph = resolver.adaptTo(SocialGraph.class);
        final ValueMap properties = resource.getValueMap();
        // get the location of the activity stream. If it is not configured, use the default.
        final Boolean outgoing = properties.get(PROP_OUTGOING_DIRECTION, Boolean.TRUE);
        this.relType = properties.get(PROP_REL_TYPE, "");
        this.direction = (outgoing) ? Direction.OUTGOING : Direction.INCOMING;
        try {
            this.userId = getUserId(properties, resolver.adaptTo(UserPropertiesManager.class));
        } catch (final RepositoryException e) {
            LOG.error("Unable to find a user to list relationships for", e);
            this.userId = null;
        }
    }

    protected String getUserId(final ValueMap properties, final UserPropertiesManager upm) throws RepositoryException {
        final String userIdFromProps = properties.get(PROP_USERID, "");
        if (!StringUtils.isEmpty(userIdFromProps)) {
            return userIdFromProps;
        }
        final String userIdSuffix =
            clientUtils.getRequest() == null ? null : clientUtils.getRequest().getRequestPathInfo().getSuffix();
        if (!StringUtils.isEmpty(userIdSuffix)) {
            if (StringUtils.endsWith(userIdSuffix, "/profile")) {
                final UserProperties up =
                    upm.getUserProperties(clientUtils.getRequest().getRequestPathInfo().getSuffixResource()
                        .adaptTo(Node.class));
                return up.getAuthorizableID();
            }
            return userIdSuffix;
        }
        return resolver.getUserID();
    }

    @Override
    public int getTotalSize() {
        getItems();
        return (socialComponents != null) ? socialComponents.size() : 0;
    }

    @Override
    public void setPagination(final CollectionPagination pagination) {

    }

    @Override
    public void setSortedOrder(final CollectionSortedOrder sortedOrder) {

    }

    @Override
    public List<Object> getItems() {
        if (socialComponents == null && graph != null && direction != null && StringUtils.isNotEmpty(userId)) {
            final Vertex node = graph.getVertex(userId);
            socialComponents = new ArrayList<Object>();

            for (final com.adobe.granite.socialgraph.Relationship relationship : node.getRelationships(direction,
                getRelationshipType())) {
                if (!relationship.isVirtual()) {
                    final Edge sr = (Edge) relationship;
                    final String id = (direction == Direction.INCOMING) ? sr.getStartId() : sr.getEndId();
                    if (id.startsWith("/")) {
                        socialComponents.add(createSocialComponent(id));
                    } else {
                        socialComponents.add(createCommunityUser(id));

                    }
                }
            }
        }
        return socialComponents;
    }

    protected String getRelationshipType() {
        return Edge.FOLLOWING_RELATIONSHIP_TYPE;
    }

    private SocialComponent createSocialComponent(final String id) {
        if (StringUtils.isNotBlank(id)) {
            final SocialComponentFactoryManager scfMgr = clientUtils.getSocialComponentFactoryManager();
            if (scfMgr != null) {
                final Resource resource = resolver.getResource(id);
                if (resource != null) {
                    final SocialComponentFactory scf = scfMgr.getSocialComponentFactory(resource);
                    if (scf != null) {
                        return scf.getSocialComponent(resource, clientUtils,
                            QueryRequestInfo.DEFAULT_QUERY_INFO_FACTORY.create());
                    }
                }
            }
        }
        return null;
    }

    private User createCommunityUser(final String id) {
        return clientUtils.getUser(id, resolver);
    }

}
