/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2015 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.graph.client.endpoint;

import java.util.HashMap;
import java.util.Map;

import javax.jcr.Session;
import javax.servlet.http.HttpServletResponse;

import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Reference;
import org.apache.jackrabbit.util.Text;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.request.RequestParameterMap;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceUtil;
import org.apache.sling.api.resource.SyntheticResource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.cq.social.graph.Edge;
import com.adobe.cq.social.graph.Vertex;
import com.adobe.cq.social.graph.client.api.Following;
import com.adobe.cq.social.graph.client.api.MutableFollowing;
import com.adobe.cq.social.scf.Operation;
import com.adobe.cq.social.scf.OperationException;
import com.adobe.cq.social.scf.OperationExtension;
import com.adobe.cq.social.scf.SocialComponent;
import com.adobe.cq.social.scf.SocialComponentFactory;
import com.adobe.cq.social.scf.SocialComponentFactoryManager;
import com.adobe.cq.social.scf.core.operations.AbstractOperationService;
import com.adobe.granite.socialgraph.SocialGraphException;

/**
 * Provides abstract implementation of the Following operation..
 * @param <T> is a {@link OperationExtension} that will be used as hooks by the extending class.
 * @param <U> is a {@link Operation} that is being provided by the extending class.
 */
@Component(metatype = false, componentAbstract = true)
public abstract class AbstractFollowingOperationService<T extends OperationExtension, U extends Operation> extends
    AbstractOperationService<T, U, Following> implements FollowingOperations {

    /**
     * Logger.
     */
    private static final Logger LOG = LoggerFactory.getLogger(AbstractFollowingOperationService.class);
    /** Social Component Factory Manager. */
    @Reference
    private SocialComponentFactoryManager componentFactoryManager;

    /**
     * Get the <code>SocialComponent</code> for the specified {@link Resource} and {@link SlingHttpServletRequest}.
     * @param comment the target following
     * @param request the client request
     * @return the {@link SocialComponent}
     */
    private Following getSocialComponent(final Resource following, final SlingHttpServletRequest request) {
        Resource resource = following;
        if (ResourceUtil.isNonExistingResource(resource)) {
            resource =
                new SyntheticResource(following.getResourceResolver(), "/social/socialgraph/"
                        + Text.getName(getResourceType()), getResourceType());
        }
        final SocialComponentFactory factory = componentFactoryManager.getSocialComponentFactory(resource);
        return (Following) ((factory != null) ? factory.getSocialComponent(following, request) : null);
    }

    @Override
    public SocialComponent follow(final SlingHttpServletRequest request) throws OperationException {
        final Resource resource = request.getResource();

        Following following;
        try {
            following = getSocialComponent(resource, request);
        } catch (final IllegalArgumentException e) {
            throw new OperationException("user information is missing", e, HttpServletResponse.SC_BAD_REQUEST);
        }
        try {
            if (following != null) {
                // TODO: There is a race window where multiple requests can come to the server and we may not
                // be able to perform atomic test and set here. Need to figure out how to deal with duplicate
                // write. Perhaps we just need to clean up the duplicate during read operation.
                if (following.getIsFollowed()) {

                    throw new OperationException(following.getUser().getName() + " is already followed "
                            + following.getFollowedId(), HttpServletResponse.SC_BAD_REQUEST);
                }

                final U followOperation = getFollowOperation();
                final Map<String, Object> requestParams = new HashMap<String, Object>();
                final RequestParameterMap params = request.getRequestParameterMap();
                for (final String key : params.keySet()) {
                    requestParams.put(key, params.getValue(key).getString());
                }
                final ResourceResolver resolver = resource.getResourceResolver();
                final Session session = resolver.adaptTo(Session.class);
                performBeforeActions(followOperation, session, resource, requestParams);

                // perform the operation
                final Vertex node = following.userNode();
                final Vertex other = following.followedNode();
                final String relType = following.getIsFollowedUser() ? "USER" : "RESOURCE";
                node.createRelationshipTo(other, getRelationshipType(request), relType);
                following.socialGraph().save();

                // extension callback with a new following state instance.
                if (following instanceof MutableFollowing) {
                    ((MutableFollowing) following).setFollowingState(true);
                } else {
                    following = getSocialComponent(resource, request);
                }
                if (following != null) {
                    performAfterActions(followOperation, session, following, requestParams);
                    return following;
                }
            }
        } catch (final SocialGraphException e) {
            throw new OperationException("Internal error creating following.", e,
                HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
        }
        // Get here if we are not able to perform the request
        throw new OperationException("Unable to obtain Following component.",
            HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
    }

    @Override
    public Following unfollow(final SlingHttpServletRequest request) throws OperationException {
        final Resource resource = request.getResource();
        Following following;
        try {
            following = getSocialComponent(resource, request);
        } catch (final IllegalArgumentException e) {
            throw new OperationException("user information is missing", e, HttpServletResponse.SC_BAD_REQUEST);
        }
        if (following != null) {
            try {
                // TODO: There is a race condition here since we can't perform atomic test and set.
                if (!following.getIsFollowed()) {
                    throw new OperationException(following.getUser().getName() + " is not followed "
                            + following.getFollowedId(), HttpServletResponse.SC_BAD_REQUEST);
                }
                final U unfollowOperation = getUnfollowOperation();
                final Map<String, Object> requestParams = new HashMap<String, Object>();
                final RequestParameterMap params = request.getRequestParameterMap();
                for (final String key : params.keySet()) {
                    requestParams.put(key, params.getValue(key).getString());
                }

                final ResourceResolver resolver = resource.getResourceResolver();
                final Session session = resolver.adaptTo(Session.class);
                performBeforeActions(unfollowOperation, session, resource, requestParams);
                final Edge edge = following.edge();
                edge.delete();
                if (following instanceof MutableFollowing) {
                    ((MutableFollowing) following).setFollowingState(false);
                } else {
                    following = getSocialComponent(resource, request);
                }
                // extension callback with a new following state instance.
                if (following != null) {
                    performAfterActions(unfollowOperation, session, following, requestParams);
                    return following;
                }
            } catch (final SocialGraphException e) {
                throw new OperationException("Failed to perform unfollow operation.", e,
                    HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
            }
        }

        throw new OperationException("Unable to obtain Following component.",
            HttpServletResponse.SC_INTERNAL_SERVER_ERROR);

    }

    protected String getRelationshipType(final SlingHttpServletRequest request) {
        return Edge.FOLLOWING_RELATIONSHIP_TYPE;
    }

    protected String getResourceType() {
        return Following.RESOURCE_TYPE;
    }

    protected abstract U getFollowOperation();

    protected abstract U getUnfollowOperation();
}