/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.group.api;

import java.security.AccessControlException;

import javax.jcr.RepositoryException;
import javax.jcr.security.AccessControlList;
import javax.jcr.security.AccessControlManager;

import org.apache.jackrabbit.api.security.user.Group;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.jcr.api.SlingRepository;

/**
 * <code>GroupService</code> provides a service managing community groups.
 */
public interface GroupService {

    static final String REPLICATE_NODE_TYPE = "oak:Unstructured";
    static final String REPLICATE_NODE_NAME = "created";
    static final String REPLICATE_PROPERTY_PATH = "path";
    static final String REPLICATE_PROPERTY_ACTION = "action";

    /**
     * Returns a user group.
     * @param repository sling repository, no longer used, may be null.
     * @param adminResolver a resource resolver with full repository access.
     * @param groupName user group name.
     * @param intermediatePath user group intermediate path.
     * @param liveCopyPath live copy site path.
     * @param liveCopyName live copy node name.
     * @param type administrative group or member group.
     * @return the user group.
     * @throws GroupException - thrown if unable to create the group.
     * @throws RepositoryException - thrown if unable to create the group.
     */
    @Deprecated
    Group createGroup(SlingRepository repository, ResourceResolver adminResolver, final String groupName,
        final String intermediatePath, final String liveCopyPath, final String liveCopyName, final String type)
        throws GroupException, RepositoryException;

    /**
     * Returns a user group.
     * @param userAdminResolver User Admin resource resolver.
     * @param msmResolver MSM resource resolver
     * @param groupName user group name.
     * @param intermediatePath user group intermediate path.
     * @param liveCopyPath live copy site path.
     * @param liveCopyName live copy node name.
     * @param type administrative group or member group.
     * @return the user group.
     * @throws GroupException - thrown if unable to create the group.
     * @throws RepositoryException - thrown if unable to create the group.
     */
    Group createGroup(ResourceResolver userAdminResolver, ResourceResolver msmResolver, String groupName,
        String intermediatePath, String liveCopyPath, String liveCopyName, String type) throws GroupException,
        RepositoryException;

    /**
     * Add an authorizable to a group.
     * @param repository sling repository.
     * @param groupName user group name.
     * @param authorizableId authorizable ID.
     * @throws GroupException - thrown if unable to add the authorizable.
     * @throws RepositoryException - thrown if unable to add the authorizable.
     */
    void addGroupMember(ResourceResolver resolver, String groupName, String authorizableId) throws GroupException,
        RepositoryException;

    /**
     * Add authorizables to a group.
     * @param repository sling repository.
     * @param groupName user group name.
     * @param authorizableIds a string array of authorizable IDs.
     * @throws GroupException - thrown if unable to add the authorizables.
     * @throws RepositoryException - thrown if unable to add the authorizables.
     */
    void addGroupMembers(ResourceResolver idResolver, String groupName, String[] authorizableIds)
        throws GroupException, RepositoryException;

    /**
     * Remove an authorizable from a group.
     * @param repository sling repository.
     * @param groupName user group name.
     * @param authorizableId authorizable ID.
     * @throws GroupException - thrown if unable to remove the authorizable.
     * @throws RepositoryException - thrown if unable to remove the authorizable.
     */
    void removeGroupMember(ResourceResolver resolver, String groupName, String authorizableId) throws GroupException,
        RepositoryException;

    /**
     * Remove authorizables from a group.
     * @param repository sling repository.
     * @param groupName user group name.
     * @param authorizableIds a string array of authorizable IDs.
     * @throws GroupException - thrown if unable to remove the authorizables.
     * @throws RepositoryException - thrown if unable to remove the authorizables.
     */
    void removeGroupMembers(ResourceResolver resolver, String groupName, String[] authorizableIds)
        throws GroupException, RepositoryException;

    /**
     * Copy Access Control List from blueprint to the community group.
     * @param resolver resource resolver.
     * @param sourcePath blueprint site path.
     * @param langs live copy languages.
     * @param destPath live copy site path.
     * @param groupId group id.
     * @param type community group privacy type.
     * @throws GroupException - thrown if unable to copy ACL.
     * @throws RepositoryException - thrown if unable to copy ACL.
     * @throws AccessControlException - thrown if unable to copy ACL.
     */
    void copyAccessControlPolicy(ResourceResolver resolver, final String sourcePath, final String langs,
        final String destPath, final String groupId, String type) throws GroupException, RepositoryException,
        AccessControlException;

    /**
     * Copy Access Control List from blueprint to the community group.
     * @param msmResolver resource resolver with read/write access to /content
     * @param userManagerResolver resource resolver with all access to users/groups and their content.
     * @param aclResolver resource resolver with read/write ACL and node type management access to /content.
     * @param sourcePath blueprint site path.
     * @param langs live copy languages.
     * @param destPath live copy site path.
     * @param groupId group id.
     * @param type community group privacy type.
     * @throws GroupException - thrown if unable to copy ACL.
     * @throws RepositoryException - thrown if unable to copy ACL.
     * @throws AccessControlException - thrown if unable to copy ACL.
     */
    void copyAccessControlPolicy(ResourceResolver msmResolver, ResourceResolver userManagerResolver,
        ResourceResolver aclResolver, String sourcePath, String langs, String destPath, String groupId, String type)
        throws GroupException, RepositoryException, AccessControlException;

    /**
     * Get access control list of the given path.
     * @param acManager Access Control Manager.
     * @param path path.
     */
    AccessControlList getACL(final AccessControlManager acManager, final String path);

}
