/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.cq.social.group.client.api;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.Binary;
import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.observation.Event;
import javax.jcr.observation.EventIterator;
import javax.jcr.observation.EventListener;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang.StringUtils;
import org.apache.felix.scr.annotations.Activate;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Deactivate;
import org.apache.felix.scr.annotations.Properties;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.Service;
import org.apache.sling.api.resource.LoginException;
import org.apache.sling.api.resource.PersistenceException;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceResolverFactory;
import org.apache.sling.api.resource.ValueMap;
import org.apache.sling.jcr.api.SlingRepository;
import org.apache.sling.jcr.resource.JcrResourceConstants;
import org.apache.sling.settings.SlingSettingsService;
import org.apache.tika.config.TikaConfig;
import org.apache.tika.detect.Detector;
import org.apache.tika.io.TikaInputStream;
import org.apache.tika.metadata.Metadata;
import org.apache.tika.metadata.TikaMetadataKeys;
import org.apache.tika.mime.MediaType;
import org.osgi.service.component.ComponentContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.cq.social.SocialException;
import com.adobe.cq.social.blueprint.api.SiteActivationService;
import com.adobe.cq.social.blueprint.api.SiteBlueprintConstants;
import com.adobe.cq.social.blueprint.api.TemplateRolloutService;
import com.adobe.cq.social.community.api.CommunityConstants;
import com.adobe.cq.social.group.api.GroupConstants;
import com.adobe.cq.social.group.api.GroupException;
import com.adobe.cq.social.scf.OperationException;
import com.adobe.cq.social.scf.SocialComponentFactoryManager;
import com.day.cq.commons.jcr.JcrUtil;
import com.day.cq.replication.ReplicationException;

//import org.apache.sling.settings.SlingSettingsService;

@Service
@Component(immediate = true, metatype = false)
@Properties({@Property(name = "service.description", value = "CommunityGroup Author Event Listener")})
public class CommunityGroupAuthorEventListener implements EventListener {

    private static final String MSM_SERVICE = "msm-service";

    private static final String USER_ADMIN = "user-admin";

    /** Logger for this class. */
    private static final Logger LOG = LoggerFactory.getLogger(CommunityGroupAuthorEventListener.class);

    private Session userAdminSession;
    private ResourceResolver userAdminResolver;

    @Reference
    private TemplateRolloutService templateRolloutService;

    @Reference
    private CommunityGroupService communityGroupService;

    @Reference
    private SiteActivationService siteActivateService;

    @Reference
    private ResourceResolverFactory resourceResolverFactory;

    @Reference
    private SlingRepository repository;

    @Reference
    private SlingSettingsService settingsService;

    @Reference
    private SocialComponentFactoryManager componentFactoryManager;

    @Reference
    private com.adobe.cq.social.serviceusers.internal.ServiceUserWrapper serviceUserWrapper;

    private static final String[] LISTEN_TO_NODE_TYPE = {SiteActivationService.REPLICATE_NODE_TYPE, "sling:Folder"};
    private static final String LISTEN_TO_NODE_PROPERTY = SiteActivationService.REPLICATE_PROPERTY_PATH;
    private static final String[] LISTEN_TO_ROOT_PATH_AUTHOR = {CommunityConstants.TENANTS_TMP_ROOT_PATH,
        CommunityConstants.COMMUNITY_TMP_ROOT_PATH + CommunityGroupConstants.ROOT_FOR_CREATE_COMMUNITY_NUGGETS};
    private static final List<String> LISTEN_TO_NODE_NAME = Arrays.asList(SiteActivationService.REPLICATE_NODE_NAME);
    private static final String EXCLUDE_NODE_NAME = "image";

    @Activate
    public void activate(final ComponentContext context) throws SocialException, OperationException {
        LOG.info("activating CommunityGroupAuthorEventListener");
        if (!isPublishMode()) {
            try {
                userAdminSession = serviceUserWrapper.loginService(repository, USER_ADMIN);
                final Map<String, Object> authInfo = new HashMap<String, Object>();
                authInfo.put(JcrResourceConstants.AUTHENTICATION_INFO_SESSION, userAdminSession);
                userAdminResolver = resourceResolverFactory.getResourceResolver(authInfo);

                // multi tenancy
                userAdminSession
                    .getWorkspace()
                    .getObservationManager()
                    .addEventListener(this, Event.NODE_ADDED /* event types */,
                        LISTEN_TO_ROOT_PATH_AUTHOR[0] /* path */, true /* isDeep */, null /* uuid */,
                        LISTEN_TO_NODE_TYPE /* nodeTypeName */, true /* noLocal */
                    );
                // OnPrem
                userAdminSession
                    .getWorkspace()
                    .getObservationManager()
                    .addEventListener(this, Event.NODE_ADDED /* event types */,
                        LISTEN_TO_ROOT_PATH_AUTHOR[1] /* path */, true /* isDeep */, null /* uuid */,
                        LISTEN_TO_NODE_TYPE /* nodeTypeName */, true /* noLocal */
                    );
            } catch (final LoginException e) {
                throw new OperationException("Cannot login", e, HttpServletResponse.SC_FORBIDDEN);
            } catch (final RepositoryException e) {
                throw new SocialException("unable to register session", e);
            }
        }

    }

    @Deactivate
    public void deactivate() {
        if (userAdminResolver != null) {
            userAdminResolver.close();
        }
        if (userAdminSession != null && userAdminSession.isLive()) {
            userAdminSession.logout();
        }
    }

    private boolean isPublishMode() {
        return settingsService != null && settingsService.getRunModes().contains("publish");
    }

    @Override
    public void onEvent(final EventIterator eventIterator) {
        while (eventIterator.hasNext()) {
            try {
                processEvent(eventIterator.nextEvent());
            } catch (final RepositoryException e) {
                LOG.error("Error while treating events", e);
            } catch (final OperationException e) {
                LOG.error("Error while treating events", e);
            } catch (final SocialException e) {
                LOG.error("Error while treating events", e);
            } catch (final PersistenceException e) {
                LOG.error("Error while treating events", e);
            }
        }
    }

    protected void processEvent(final Event event) throws SocialException, RepositoryException, OperationException,
        PersistenceException {
        LOG.info("process site publish event at : {}", event.getPath());
        final String payload = event.getPath();
        if (StringUtils.endsWith(payload, EXCLUDE_NODE_NAME))
            return;

        final Resource resource = userAdminResolver.getResource(payload);
        if (resource == null) {
            throw new SocialException("unable to find event payload " + payload);
        }
        final String path = resource.adaptTo(ValueMap.class).get(LISTEN_TO_NODE_PROPERTY, String.class);

        if (LISTEN_TO_NODE_NAME.contains(resource.getName())
                && resource.getResourceType().equals(LISTEN_TO_NODE_TYPE[0])) {
            if ((payload.startsWith(LISTEN_TO_ROOT_PATH_AUTHOR[0]) && payload
                .contains(CommunityGroupConstants.ROOT_FOR_CREATE_COMMUNITY_NUGGETS))
                    || payload.startsWith(LISTEN_TO_ROOT_PATH_AUTHOR[1])) {
                final ValueMap properties = resource.adaptTo(ValueMap.class);
                // create community group event
                if (CommunityGroupConstants.ACTION_TYPE_CREATE_COMMUNITY_GROUP.equals(properties
                    .get(SiteActivationService.REPLICATE_PROPERTY_ACTION))) {
                    createCommunityGroup(path, properties);
                    userAdminResolver.delete(resource.getParent());
                    userAdminSession.save();
                }
            }
        }

    }

    private void createCommunityGroup(final String groupsRoot, final ValueMap properties) throws SocialException,
        RepositoryException, OperationException {
        if (StringUtils.isEmpty(groupsRoot)) {
            throw new SocialException("Node property " + LISTEN_TO_NODE_PROPERTY + " is missing. ");
        }
        if (communityGroupService == null) {
            throw new SocialException("Failed to get CommunityGroupService.");
        }
        if (siteActivateService == null) {
            throw new SocialException("Failed to get GroupService.");
        }

        ResourceResolver msmResolver = null;
        ResourceResolver userAdminResolver = null;
        try {
            msmResolver =
                serviceUserWrapper.getServiceResourceResolver(resourceResolverFactory,
                    Collections.singletonMap(ResourceResolverFactory.SUBSERVICE, (Object) MSM_SERVICE));
            final Session session = msmResolver.adaptTo(Session.class);
            final String liveCopyName =
                properties.get(CommunityGroupConstants.PROP_COMMUNITY_GROUP_NAME, String.class);
            final String liveCopyTitle =
                properties.get(CommunityGroupConstants.PROP_COMMUNITY_GROUP_TITLE, liveCopyName);
            final String bluePrintPath =
                properties.get(CommunityGroupConstants.PROP_COMMUNITY_GROUP_BLUEPRINT_ID, String.class);

            try {
                // roll out pages
                final String pagePath =
                    templateRolloutService.rolloutBlueprint(session, bluePrintPath, liveCopyName, liveCopyTitle,
                        groupsRoot);
                final Resource groupPage = msmResolver.getResource(pagePath);

                // copy image
                userAdminResolver =
                    serviceUserWrapper.getServiceResourceResolver(resourceResolverFactory,
                        Collections.singletonMap(ResourceResolverFactory.SUBSERVICE, (Object) USER_ADMIN));
                final Resource image =
                    userAdminResolver.getResource(properties.get(SiteBlueprintConstants.PROPERTY_FORM_PAYLOAD) + "/"
                            + GroupConstants.PROPERTY_IMAGE_NAME);
                if (image != null) {
                    // sanitize image attachment
                    final Node imageNode = image.adaptTo(Node.class);
                    final String mimeType = getResourceMIMEType(image);
                    if (StringUtils.startsWithIgnoreCase(mimeType, "image")) {
                        final Node imageCopy =
                            JcrUtil.createPath(pagePath + "/" + SiteBlueprintConstants.PROPERTY_IMAGE_PATH,
                                "sling:Folder", userAdminResolver.adaptTo(Session.class));
                        JcrUtil.copy(imageNode, imageCopy, SiteBlueprintConstants.PROPERTY_IMAGE_NAME);
                        try {
                            userAdminResolver.commit();
                        } catch (PersistenceException e) {
                            throw new SocialException("Failed creating a community group", e);
                        }
                    } else {
                        LOG.error("upload invalid image type {}", mimeType);
                    }
                }

                // set configure node
                communityGroupService.setCommunityGroupConfigure(msmResolver, pagePath, properties);

                // create user groups and apply ACLs user user-admin resolver, but groupPage is from msmResolver.
                communityGroupService.prepareUserGroups(userAdminResolver, groupPage, groupsRoot, properties);

                // publish community group, this uses the msmResolver for access to /content.
                siteActivateService.activateCommunity(userAdminResolver, msmResolver, pagePath);

            } catch (final GroupException e) {
                throw new SocialException("Failed to create community group " + liveCopyName + " at " + groupsRoot, e);
            } catch (final ReplicationException e) {
                throw new SocialException(
                    "Failed to replicate community group " + liveCopyName + " at " + groupsRoot, e);
            }
        } catch (final LoginException e) {
            throw new SocialException("Failed creating a community gropup", e);
        } finally {
            if (msmResolver != null) {
                try {
                    msmResolver.commit();
                } catch (final PersistenceException e) {
                    throw new SocialException("Failed creating a community group", e);
                } finally {
                    msmResolver.close();
                }
            }
            if (userAdminResolver != null) {
                userAdminResolver.close();
            }
        }
    }

    /**
     * Utility to get MIME type of the resource.
     * @param res Resource
     * @return resource MIME type
     * @throws RepositoryException - thrown if resource content metadata does not exist
     */
    private String getResourceMIMEType(final Resource res) throws RepositoryException {
        final Node node = res.adaptTo(Node.class);
        final Node ntResourceNode = node.getNode("jcr:content");

        try {
            final Binary bin = ntResourceNode.getProperty("jcr:data").getBinary();
            final TikaConfig config = TikaConfig.getDefaultConfig();
            final Detector detector = config.getDetector();

            final TikaInputStream stream = TikaInputStream.get(bin.getStream());

            final Metadata metadata = new Metadata();
            metadata.add(TikaMetadataKeys.RESOURCE_NAME_KEY, "image");
            final MediaType mediaType = detector.detect(stream, metadata);
            return mediaType.getType();
        } catch (final PathNotFoundException el) {
            return null;
        } catch (final IOException e) {
            return ntResourceNode.getProperty("jcr:mimeType").getString();
        }

    }
}