/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.cq.social.group.client.api;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.observation.Event;
import javax.jcr.observation.EventIterator;
import javax.jcr.observation.EventListener;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang.StringUtils;
import org.apache.felix.scr.annotations.Activate;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Deactivate;
import org.apache.felix.scr.annotations.Properties;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.Service;
import org.apache.sling.api.resource.LoginException;
import org.apache.sling.api.resource.ModifiableValueMap;
import org.apache.sling.api.resource.PersistenceException;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceResolverFactory;
import org.apache.sling.api.resource.ValueMap;
import org.apache.sling.jcr.api.SlingRepository;
import org.apache.sling.jcr.resource.JcrResourceConstants;
import org.apache.sling.settings.SlingSettingsService;
import org.osgi.service.component.ComponentContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.cq.social.SocialException;
import com.adobe.cq.social.blueprint.api.SiteActivationService;
import com.adobe.cq.social.community.api.CommunityConstants;
import com.adobe.cq.social.console.utils.JCRUtils;
import com.adobe.cq.social.group.api.GroupConstants;
import com.adobe.cq.social.group.api.GroupException;
import com.adobe.cq.social.scf.OperationException;
import com.adobe.cq.social.scf.SocialComponentFactoryManager;
import com.adobe.cq.social.serviceusers.internal.ServiceUserWrapper;
import com.day.cq.commons.jcr.JcrUtil;

//import org.apache.sling.settings.SlingSettingsService;

@Service
@Component(immediate = true, metatype = false)
@Properties({@Property(name = "service.description", value = "CommunityGroup Publish Event Listener"),})
public class CommunityGroupPublishEventListener implements EventListener {

    private static final String MSM_SERVICE = "msm-service";

    private static final String USER_ADMIN = "user-admin";

    private static final String TYPE_OAK_UNSTRUCTURED = "oak:Unstructured";
    private static final String SLING_RESOURCETYPE = "sling:resourceType";
    private static final String SUBSCRIPTIONS_NODE_NAME = "subscriptions";
    private static final String SUBSCRIPTIONS_RESOURCE_TYPE = "social/subscriptions/components/hbs/subscriptions";

    /** Logger for this class. */
    private static final Logger LOG = LoggerFactory.getLogger(CommunityGroupPublishEventListener.class);

    private Session userAdminSession;
    private ResourceResolver userAdminResolver;

    @Reference
    private CommunityGroupService communityGroupService;

    @Reference
    private ResourceResolverFactory resourceResolverFactory;

    @Reference
    private SlingRepository repository;

    @Reference
    private SlingSettingsService settingsService;

    @Reference
    private SocialComponentFactoryManager componentFactoryManager;

    @Reference
    private ServiceUserWrapper serviceUserWrapper;

    private static final String[] LISTEN_TO_NODE_TYPE = {SiteActivationService.REPLICATE_NODE_TYPE};
    private static final String LISTEN_TO_NODE_PROPERTY = SiteActivationService.REPLICATE_PROPERTY_PATH;
    private static final String[] LISTEN_TO_ROOT_PATH_PUBLISH = {CommunityConstants.TENANTS_TMP_ROOT_PATH,
        CommunityConstants.COMMUNITY_TMP_ROOT_PATH + CommunityGroupConstants.ROOT_FOR_PUBLISH_COMMUNITY_NUGGETS};
    private final static List<String> LISTEN_TO_NODE_NAME = Arrays.asList(SiteActivationService.REPLICATE_NODE_NAME);

    // delimiters to split the invited id list
    private final String delimiters = "[\\s,;]+";

    @Activate
    public void activate(final ComponentContext context) throws SocialException, OperationException {
        LOG.info("activating CommunityGroupPublishEventListener");
        if (isPublishMode()) {
            try {
                userAdminSession = serviceUserWrapper.loginService(repository, USER_ADMIN);
                final Map<String, Object> authInfo = new HashMap<String, Object>();
                authInfo.put(JcrResourceConstants.AUTHENTICATION_INFO_SESSION, userAdminSession);
                userAdminResolver = resourceResolverFactory.getResourceResolver(authInfo);

                userAdminSession
                    .getWorkspace()
                    .getObservationManager()
                    .addEventListener(this, Event.NODE_ADDED /* event types */,
                        LISTEN_TO_ROOT_PATH_PUBLISH[0] /* path */, true /* isDeep */, null /* uuid */,
                        LISTEN_TO_NODE_TYPE /* nodeTypeName */, true /* noLocal */
                    );
                userAdminSession
                    .getWorkspace()
                    .getObservationManager()
                    .addEventListener(this, Event.NODE_ADDED /* event types */,
                        LISTEN_TO_ROOT_PATH_PUBLISH[1] /* path */, true /* isDeep */, null /* uuid */,
                        LISTEN_TO_NODE_TYPE /* nodeTypeName */, true /* noLocal */
                    );
            } catch (final LoginException e) {
                throw new OperationException("Cannot login", e, HttpServletResponse.SC_FORBIDDEN);
            } catch (final RepositoryException e) {
                throw new SocialException("unable to register session", e);
            }
        }

    }

    @Deactivate
    public void deactivate() {
        if (userAdminResolver != null) {
            userAdminResolver.close();
        }
        if (userAdminSession != null) {
            userAdminSession.logout();
        }
    }

    private boolean isPublishMode() {
        return settingsService != null && settingsService.getRunModes().contains("publish");
    }

    @Override
    public void onEvent(final EventIterator eventIterator) {
        while (eventIterator.hasNext()) {
            try {
                processEvent(eventIterator.nextEvent());
            } catch (final RepositoryException e) {
                LOG.error("Error while treating events", e);
            } catch (final OperationException e) {
                LOG.error("Error while treating events", e);
            } catch (final SocialException e) {
                LOG.error("Error while treating events", e);
            } catch (final PersistenceException e) {
                LOG.error("Error while treating events", e);
            }
        }
    }

    protected void processEvent(final Event event) throws SocialException, RepositoryException, OperationException,
        PersistenceException {
        LOG.info("process site publish event at : {}", event.getPath());
        final String payload = event.getPath();

        final Resource resource = userAdminResolver.getResource(payload);
        if (resource == null) {
            throw new SocialException("unable to find event payload " + payload);
        }

        if (LISTEN_TO_NODE_NAME.contains(resource.getName())
                && resource.getResourceType().equals(LISTEN_TO_NODE_TYPE[0])) {
            if ((payload.startsWith(LISTEN_TO_ROOT_PATH_PUBLISH[0]) && payload
                .contains(CommunityGroupConstants.ROOT_FOR_PUBLISH_COMMUNITY_NUGGETS))
                    || payload.startsWith(LISTEN_TO_ROOT_PATH_PUBLISH[1])) {
                final ValueMap properties = resource.adaptTo(ValueMap.class);
                // publish community group event
                if (CommunityGroupConstants.ACTION_TYPE_PUBLISH_COMMUNITY_GROUP.equals(properties
                    .get(SiteActivationService.REPLICATE_PROPERTY_ACTION))) {
                    final String path = resource.adaptTo(ValueMap.class).get(LISTEN_TO_NODE_PROPERTY, String.class);
                    publishCommunityGroup(path, properties);
                    userAdminResolver.delete(resource.getParent());
                    userAdminSession.save();
                }
            }
        }

    }

    private void publishCommunityGroup(final String group, final ValueMap properties) throws SocialException,
        RepositoryException, OperationException {
        if (StringUtils.isEmpty(group)) {
            throw new SocialException("Node property " + LISTEN_TO_NODE_PROPERTY + " is missing. ");
        }
        if (communityGroupService == null) {
            throw new SocialException("Failed to get CommunityGroupService.");
        }

        final String liveCopyName = properties.get(CommunityGroupConstants.PROP_COMMUNITY_GROUP_NAME, String.class);

        ResourceResolver msmResolver = null;
        try {
            msmResolver =
                serviceUserWrapper.getServiceResourceResolver(resourceResolverFactory,
                    Collections.singletonMap(ResourceResolverFactory.SUBSERVICE, (Object) MSM_SERVICE));
            // create user groups
            final Resource resource = msmResolver.getResource(group);
            final String groupsRoot = resource.getParent().getPath();
            // create user groups and apply ACLs
            // resolver is user admin resolver, resource is from the msmResolver.
            communityGroupService.prepareUserGroups(userAdminResolver, resource, groupsRoot, properties);

            // create subscriptions node
            final Node node = resource.adaptTo(Node.class);
            if (node != null) {
                final Node subNode = node.addNode(SUBSCRIPTIONS_NODE_NAME, TYPE_OAK_UNSTRUCTURED);
                subNode.setProperty(SLING_RESOURCETYPE, SUBSCRIPTIONS_RESOURCE_TYPE);
            }

            // clean up
            final ModifiableValueMap values =
                resource.getChild(CommunityGroupConstants.CONFIG_NODE_NAME).adaptTo(ModifiableValueMap.class);
            final String payload = values.get(GroupConstants.PROPERTY_FORM_PAYLOAD, String.class);
            if (!StringUtils.isEmpty(payload)) {
                final Resource formPayload = userAdminResolver.getResource(payload);
                if (formPayload != null) {
                    userAdminResolver.delete(formPayload);
                }
                values.remove(GroupConstants.PROPERTY_FORM_PAYLOAD);
            }

            userAdminResolver.adaptTo(Session.class).save();
            // deliberately not performing any of the form payload property removals if group management failed.
            msmResolver.adaptTo(Session.class).save();
        } catch (final GroupException e) {
            throw new SocialException("Failed to create community group " + liveCopyName + " at " + group, e);
        } catch (final PersistenceException e) {
            LOG.error("Failed to clean up form payload " + liveCopyName + " at " + group, e);
        } catch (final LoginException e) {
            throw new SocialException("Failed to acquire service user resolver while preparing"
                    + "community group user groups");
        } finally {
            if (msmResolver != null) {
                msmResolver.close();
            }
        }
    }

}