/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.messaging.client.api;

import javax.annotation.Nonnull;

import org.apache.sling.api.resource.PersistenceException;

/**
 * Represents counts of messages in a message box.
 */
public interface MessageCounts {

    /**
     * @return count of not trashed unread messages.
     */
    long getNonDeletedUnreadCount();

    /**
     * @return count of not trashed read messages.
     */
    long getNonDeletedReadCount();

    /**
     * @return count of trashed unread messages.
     */
    long getDeletedUnreadCount();

    /**
     * @return count of trashed+read messages.
     */
    long getDeletedReadCount();

    /**
     * Get the count type specified by the parameter.
     * @param type The count type to return.
     * @return The count.
     */
    long getCount(CountType type);

    /**
     * Get the total count of messages in the message box.
     * @return The total count.
     */
    long getTotalCount();

    /**
     * Increment a count property by 1.
     * @param type The {@link MessageCounts.CountType}
     * @throws PersistenceException If the increment fails.
     */
    void incrementCount(@Nonnull final MessageCounts.CountType type) throws PersistenceException;

    /**
     * Decrement a count proprety by 1.
     * @param type The {@link MessageCounts.CountType}
     * @throws PersistenceException If the decrement fails.
     */
    void decrementCount(@Nonnull final MessageCounts.CountType type) throws PersistenceException;

    /**
     * Get the total size of the message box.
     * @return The total size of the message box.
     */
    long getTotalSize();

    /**
     * Increment the value of the total size of the message box. Passing in a negative value performs a decrement.
     * @param incrementBy The amount to increment by.
     * @throws PersistenceException If the increment fails.
     */
    void incrementTotalSize(long incrementBy) throws PersistenceException;

    /**
     * Decrement value of the total size of the message box.
     * @param decrementBy The amount by which to decrement the total size of the message box.
     * @throws PersistenceException If the decrement fails.
     */
    void decrementTotalSize(long decrementBy) throws PersistenceException;

    /**
     * The count type.
     */
    enum CountType {
        /**
         * count of unread messages which have been marked as deleted.
         */
        DeletedUnreadMessageCount {
            /**
             * Converts the enum option to a {@link java.lang.String}.
             * @return The string value of the enum option.
             */
            @Override
            @Nonnull
            public String toString() {
                return "deletedUnreadMessageCount";
            }
        },
        /**
         * count of unread messages which have not been marked as deleted.
         */
        NonDeletedUnreadMessageCount {
            /**
             * Converts the enum option to a {@link java.lang.String}.
             * @return The string value of the enum option.
             */
            @Override
            @Nonnull
            public String toString() {
                return "nonDeletedUnreadMessageCount";
            }
        },
        /**
         * count of read messages which have been marked as deleted.
         */
        DeletedReadMessageCount {
            /**
             * Converts the enum option to a {@link java.lang.String}.
             * @return The string value of the enum option.
             */
            @Override
            @Nonnull
            public String toString() {
                return "deletedReadMessageCount";
            }
        },
        /**
         * count of read messages which have not been marked as deleted.
         */
        NonDeletedReadMessageCount {
            /**
             * Converts the enum option to a {@link java.lang.String}.
             * @return The string value of the enum option.
             */
            @Override
            @Nonnull
            public String toString() {
                return "nonDeletedReadMessageCount";
            }
        },
        /**
         * count of all the messages present irrespective of read and deleted flags.
         */
        TotalCount
    }
}
