/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.moderation.dashboard.api;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Dictionary;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.ReferenceCardinality;
import org.apache.felix.scr.annotations.ReferencePolicy;
import org.apache.felix.scr.annotations.Service;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.resource.Resource;
import org.osgi.service.cm.ConfigurationAdmin;
import org.osgi.service.component.ComponentContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.cq.social.moderation.dashboard.impl.FilterGroupImpl;
import com.adobe.cq.social.scf.ClientUtilities;
import com.adobe.cq.social.scf.QueryRequestInfo;
import com.adobe.cq.social.scf.SocialComponent;
import com.adobe.cq.social.scf.core.AbstractSocialComponentFactory;

@Component(metatype = true, label = "AEM Communities Moderation Dashboard Filters",
        description = "This component holds a group" + " of search filters.", immediate = true)
@Service
public class FilterGroupSocialComponentFactory extends AbstractSocialComponentFactory {

    /** Lookup used to find configuration properties. */
    public static final String SERVICE_PID =
        "com.adobe.cq.social.moderation.dashboard.api.FilterGroupSocialComponentFactory";

    /** Default resourceTypeNames. */
    public static final String DEFAULT_QNA_QUESTION = "QnA Question";
    public static final String DEFAULT_QNA_ANSWER = "QnA Answer";
    public static final String DEFAULT_FORUM_TOPIC = "Forum Topic";
    public static final String DEFAULT_FORUM_REPLY = "Forum Reply";
    public static final String DEFAULT_COMMENT = "Comment";
    public static final String DEFAULT_FILE = "File Library Document";
    public static final String DEFAULT_FOLDER = "File Library Folder";
    public static final String DEFAULT_BLOGS = "Blog Article";
    public static final String DEFAULT_BLOGS_COMMENTS = "Blog Comment";
    public static final String DEFAULT_CALENDAR_EVENTS = "Calendar Event";
    public static final String DEFAULT_CALENDAR_EVENT_COMMENTS = "Calendar Comment";

    /** Default resourceTypes. */
    public static final String DEFAULT_QNA_QUESTION_RT = "social/qna/components/topic";
    public static final String DEFAULT_QNA_QUESTION_HBS_RT = "social/qna/components/hbs/topic";
    public static final String DEFAULT_QNA_ANSWER_RT = "social/qna/components/post";
    public static final String DEFAULT_QNA_ANSWER_HBS_RT = "social/qna/components/hbs/post";
    public static final String DEFAULT_FORUM_TOPIC_RT = "social/forum/components/topic";
    public static final String DEFAULT_FORUM_TOPIC_HBS_RT = "social/forum/components/hbs/topic";
    public static final String DEFAULT_FORUM_REPLY_RT = "social/forum/components/post";
    public static final String DEFAULT_FORUM_REPLY_HBS_RT = "social/forum/components/hbs/post";
    public static final String DEFAULT_COMMENT_RT = "social/commons/components/comments/comment";
    public static final String DEFAULT_COMMENT_HBS_RT = "social/commons/components/hbs/comments/comment";
    public static final String DEFAULT_FILE_HBS_RT = "social/filelibrary/components/hbs/document";
    public static final String DEFAULT_FOLDER_HBS_RT = "social/filelibrary/components/hbs/folder";
    public static final String DEFAULT_BLOGS_HBS_RT = "social/journal/components/hbs/entry_topic";
    public static final String DEFAULT_BLOGS_COMMENTS_HBS_RT = "social/journal/components/hbs/comment";
    public static final String DEFAULT_CALENDAR_EVENTS_HBS_RT = "social/calendar/components/hbs/event";
    public static final String DEFAULT_CALENDAR_EVENT_COMMENTS_HBS_RT =
        "social/calendar/components/hbs/event_comment";

    /** Name used for OSGI configuration of resourceTypes. */
    @Property(label = "ResourceType Filters", value = {"", ""},
            description = "Specify resource types for searching: ScreenName=resource/type1:resource/type2... etc.")
    public static final String RT_FILTERS = "resourceType.filters";

    /** Pattern for parsing configuration properties. */
    public static final String CONFIG_PARSE_PATTERN = "=|:";

    /** Prefix for parsing filterGroup configuration properties. */
    public static final String CONFIG_FILTER_GROUP_PREFIX = "filterGroup.";

    /** Default contentTypes map, only in case the configuration lookup fails. */
    public static final Map<String, List<String>> DEFAULT_CONTENT_TYPES_MAP =
        new LinkedHashMap<String, List<String>>(5);
    static {
        DEFAULT_CONTENT_TYPES_MAP.put(DEFAULT_COMMENT, Arrays.asList(DEFAULT_COMMENT_RT, DEFAULT_COMMENT_HBS_RT));
        DEFAULT_CONTENT_TYPES_MAP.put(DEFAULT_FORUM_TOPIC,
            Arrays.asList(DEFAULT_FORUM_TOPIC_RT, DEFAULT_FORUM_TOPIC_HBS_RT));
        DEFAULT_CONTENT_TYPES_MAP.put(DEFAULT_FORUM_REPLY,
            Arrays.asList(DEFAULT_FORUM_REPLY_RT, DEFAULT_FORUM_REPLY_HBS_RT));
        DEFAULT_CONTENT_TYPES_MAP.put(DEFAULT_QNA_QUESTION,
            Arrays.asList(DEFAULT_QNA_QUESTION_RT, DEFAULT_QNA_QUESTION_HBS_RT));
        DEFAULT_CONTENT_TYPES_MAP.put(DEFAULT_QNA_ANSWER,
            Arrays.asList(DEFAULT_QNA_ANSWER_RT, DEFAULT_QNA_ANSWER_HBS_RT));
        DEFAULT_CONTENT_TYPES_MAP.put(DEFAULT_BLOGS, Arrays.asList(DEFAULT_BLOGS_HBS_RT));
        DEFAULT_CONTENT_TYPES_MAP.put(DEFAULT_BLOGS_COMMENTS, Arrays.asList(DEFAULT_BLOGS_COMMENTS_HBS_RT));
        DEFAULT_CONTENT_TYPES_MAP.put(DEFAULT_CALENDAR_EVENTS, Arrays.asList(DEFAULT_CALENDAR_EVENTS_HBS_RT));
        DEFAULT_CONTENT_TYPES_MAP.put(DEFAULT_CALENDAR_EVENT_COMMENTS,
            Arrays.asList(DEFAULT_CALENDAR_EVENT_COMMENTS_HBS_RT));
        DEFAULT_CONTENT_TYPES_MAP.put(DEFAULT_FOLDER, Arrays.asList(DEFAULT_FOLDER_HBS_RT));
        DEFAULT_CONTENT_TYPES_MAP.put(DEFAULT_FILE, Arrays.asList(DEFAULT_FILE_HBS_RT));
    }

    /** Logger. */
    private static final Logger LOG = LoggerFactory.getLogger(FilterGroupSocialComponentFactory.class);

    /** Config admin to read resource types from config. */
    @Reference(cardinality = ReferenceCardinality.MANDATORY_UNARY, policy = ReferencePolicy.STATIC)
    private ConfigurationAdmin configurationAdmin;

    /** List of resourceType filters. */
    private Map<String, List<String>> resourceTypeFilters;

    @Override
    public SocialComponent getSocialComponent(final Resource resource) {
        return new FilterGroupImpl(resource, getClientUtilities(resource.getResourceResolver()),
            getResourceTypeFilterNames());
    }

    @Override
    public SocialComponent getSocialComponent(final Resource resource, final SlingHttpServletRequest request) {
        return new FilterGroupImpl(resource, getClientUtilities(request), getResourceTypeFilterNames());
    }

    @Override
    public SocialComponent getSocialComponent(final Resource resource, final ClientUtilities clientUtils,
        final QueryRequestInfo queryRequestInfo) {
        return new FilterGroupImpl(resource, clientUtils, getResourceTypeFilterNames());
    }

    @Override
    public String getSupportedResourceType() {
        return FilterGroup.RESOURCE_TYPE;
    }

    /**
     * Return the resourceType filter names in a List. This method can be used by other classes that need access to
     * the same data.
     * @return a List of resourceType filter names
     */
    private List<String> getResourceTypeFilterNames() {
        if (resourceTypeFilters == null) {
            setResourceTypeFilters(null);
        }

        return new ArrayList<String>(resourceTypeFilters.keySet());
    }

    /**
     * Return the resourceType filters in a Map. This method can be used by other classes that need access to the same
     * data.
     * @return a List of resourceType filter names
     */
    public Map<String, List<String>> getResourceTypeFilters() {
        if (resourceTypeFilters == null) {
            setResourceTypeFilters(null);
        }
        return resourceTypeFilters;
    }

    /**
     * Re-create the resourceTypeFilters map and pull the values from the OSGI config.
     * @param ComponentContext context Used when the component is activated. Can be null and the configuration will be
     *            taken using the configurationAdmin.
     * @return a List of resourceType filter names
     */
    private void setResourceTypeFilters(final ComponentContext context) {
        resourceTypeFilters = new LinkedHashMap<String, List<String>>();
        Dictionary<?, ?> configProperties;
        try {
            if (context == null) {
                org.osgi.service.cm.Configuration configuration = configurationAdmin.getConfiguration(SERVICE_PID);
                configProperties = configuration.getProperties();
            } else {
                configProperties = context.getProperties();
            }

            String[] rawResourceTypes = (String[]) configProperties.get(RT_FILTERS);
            if (resourceTypeFilters.size() == 0) {
                LOG.info("No resourceType filters from configuration found.  Default filters will be used.");
            }
            if (rawResourceTypes != null && rawResourceTypes.length != 0) {
                for (String resourceTypeEntry : rawResourceTypes) {
                    String[] filterEntry = resourceTypeEntry.split(CONFIG_PARSE_PATTERN);
                    // Must be at least 2 entries in string, one screen name string and one resource type string
                    if (filterEntry == null || filterEntry.length < 2) {
                        LOG.error(
                            "Resource types in configuration must contain a name and at least one resource type."
                                    + " [{}]", resourceTypeEntry);
                    } else {
                        final List<String> types = new ArrayList<String>(5);
                        types.addAll(Arrays.asList(Arrays.copyOfRange(filterEntry, 1, filterEntry.length)));
                        if (DEFAULT_CONTENT_TYPES_MAP.containsKey(filterEntry[0])) {
                            types.addAll(DEFAULT_CONTENT_TYPES_MAP.get(filterEntry[0]));
                        }
                        resourceTypeFilters.put(filterEntry[0], types);
                    }
                }
            }
        } catch (final IOException e) {
            LOG.error("Unable to read resourceType filters from configuration.", e);
        } catch (final NullPointerException npe) {
            LOG.error("Unable to read resourceType filters from configuration.", npe);
        }

        // append default list of resource types to custom configured list
        for (Entry<String, List<String>> defaultType : DEFAULT_CONTENT_TYPES_MAP.entrySet()) {
            final String name = defaultType.getKey();
            final List<String> types = defaultType.getValue();
            if (!resourceTypeFilters.containsKey(name)) {
                resourceTypeFilters.put(name, types);
            }
        }
    }

    /**
     * OSGI Activate.
     * @param context the ComponentContext sent by OSGI.
     */
    protected void activate(final ComponentContext context) {
        setResourceTypeFilters(context);
    }
}
